# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
# Authors: Eric Casteleijn <eric.casteleijn@canonical.com>

"""Test cases for field mapping"""

import copy, doctest
from testtools import TestCase
from desktopcouch.records.field_registry import (
    SimpleFieldMapping, MergeableListFieldMapping, Transformer)
from desktopcouch.records.record import Record

TEST_RECORD = {
    'record_type': 'http://example.com/test',
    'simple_field': 23,
    'test_fields': {
        'e47455fb-da05-481e-a2c7-88f14d5cc163':
            {'the_field': 'the value',
             'not_the_field': 'different value'},
        'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3': {
            'the field': 'another value'}},
    'application_annotations': {'Test App': {'private_application_annotations':{
        'test_field': 'e47455fb-da05-481e-a2c7-88f14d5cc163'}}}}

APP_RECORD = {
    'record_type': 'http://example.com/test',
    'simpleField': 23,
    'strawberryField': 'the value',}

field_registry = {
    'simpleField': SimpleFieldMapping('simple_field'),
    'strawberryField': MergeableListFieldMapping(
        'Test App', 'test_field', 'test_fields', 'the_field'),}


class AppTransformer(Transformer):
    """A test transformer class."""

    def __init__(self):
        super(AppTransformer, self).__init__('Test App', field_registry)


class TestFieldMapping(TestCase):
    """Test Case for FieldMapping objects."""

    def setUp(self):
        self.test_record = copy.deepcopy(TEST_RECORD)

    def test_simple_field_mapping(self):
        """Test the simple field mapping object."""
        r = Record(self.test_record)
        mapping = SimpleFieldMapping('simple_field')
        self.assertEqual(23, mapping.getValue(r))
        mapping.setValue(r, 'Fnord')
        self.assertEqual('Fnord', mapping.getValue(r))

    def test_mergeable_list_field_mapping(self):
        """Test the MergeableListFieldMapping object."""
        record = Record(self.test_record)
        mapping = MergeableListFieldMapping(
            'Test App', 'test_field', 'test_fields', 'the_field')
        self.assertEqual('the value', mapping.getValue(record))
        mapping.setValue(record, 'a new value')
        self.assertEqual('a new value', mapping.getValue(record))

    def test_mergeable_list_field_mapping_empty_field(self):
        """Test setting empty values in the MergeableListFieldMapping object."""
        record = Record(self.test_record)
        mapping = MergeableListFieldMapping(
            'Test App', 'test_field', 'test_fields', 'the_field')
        mapping.setValue(record, '')
        self.assertEqual(None, mapping.getValue(record))


class TestTransformer(TestCase):
    """Test application specific transformer classes"""

    def setUp(self):
        """setup test fixtures"""
        self.transformer = AppTransformer()

    def test_from_app(self):
        """Test transformation from app to Ubuntu One."""
        record = Record(record_type="http://example.com/test")
        self.transformer.from_app(APP_RECORD, record)
        underlying = record._data
        self.assertEqual(23, record['simple_field'])
        the_uuid = record.application_annotations['Test App']\
                   ['private_application_annotations']['test_field']
        self.assertEqual(
            {'the_field': 'the value'},
            underlying['test_fields'][the_uuid])

    def test_to_app(self):
        """Test transformation to app from Ubuntu One."""
        record = Record(TEST_RECORD)
        data = {}
        self.transformer.to_app(record, data)
        self.assertEqual(
            {'simpleField': 23, 'strawberryField': 'the value'}, data)

    def test_run_doctests(self):
        results = doctest.testfile('../doc/field_registry.txt')
        self.assertEqual(0, results.failed)
