# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors: Eric Casteleijn <eric.casteleijn@canonical.com>
#          Mark G. Saye <mark.saye@canonical.com>
#          Stuart Langridge <stuart.langridge@canonical.com>
#          Chad Miller <chad.miller@canonical.com>

"""The Desktop Couch Records API."""

from couchdb import Server
from couchdb.client import ResourceNotFound, ResourceConflict
from couchdb.design import ViewDefinition
from record import Record
import httplib2
from oauth import oauth
import urlparse
import cgi

#DEFAULT_DESIGN_DOCUMENT = "design"
DEFAULT_DESIGN_DOCUMENT = None  # each view in its own eponymous design doc.


class NoSuchDatabase(Exception):
    "Exception for trying to use a non-existent database"

    def __init__(self, dbname):
        self.database = dbname
        super(NoSuchDatabase, self).__init__()

    def __str__(self):
        return ("Database %s does not exist on this server. (Create it by "
                "passing create=True)") % self.database

class OAuthAuthentication(httplib2.Authentication):
    """An httplib2.Authentication subclass for OAuth"""
    def __init__(self, oauth_data, host, request_uri, headers, response, 
        content, http, scheme):
        self.oauth_data = oauth_data
        self.scheme = scheme
        httplib2.Authentication.__init__(self, None, host, request_uri, 
              headers, response, content, http)

    def request(self, method, request_uri, headers, content):
        """Modify the request headers to add the appropriate
        Authorization header."""
        consumer = oauth.OAuthConsumer(self.oauth_data['consumer_key'], 
           self.oauth_data['consumer_secret'])
        access_token = oauth.OAuthToken(self.oauth_data['token'], 
           self.oauth_data['token_secret'])
        sig_method = oauth.OAuthSignatureMethod_HMAC_SHA1
        full_http_url = "%s://%s%s" % (self.scheme, self.host, request_uri)
        schema, netloc, path, params, query, fragment = \
                urlparse.urlparse(full_http_url)
        querystr_as_dict = dict(cgi.parse_qsl(query))
        req = oauth.OAuthRequest.from_consumer_and_token(
            consumer,
            access_token,
            http_method = method,
            http_url = full_http_url,
            parameters = querystr_as_dict
        )
        req.sign_request(sig_method(), consumer, access_token)
        headers.update(httplib2._normalize_headers(req.to_header()))

class OAuthCapableHttp(httplib2.Http):
    """Subclass of httplib2.Http which specifically uses our OAuth 
       Authentication subclass (because httplib2 doesn't know about it)"""
    def __init__(self, scheme="http", cache=None, timeout=None, proxy_info=None):
        self.__scheme = scheme
        super(OAuthCapableHttp, self).__init__(cache, timeout, proxy_info)

    def add_oauth_tokens(self, consumer_key, consumer_secret, 
                         token, token_secret):
        self.oauth_data = {
            "consumer_key": consumer_key,
            "consumer_secret": consumer_secret,
            "token": token,
            "token_secret": token_secret
        }

    def _auth_from_challenge(self, host, request_uri, headers, response,
            content):
        """Since we know we're talking to desktopcouch, and we know that it
           requires OAuth, just return the OAuthAuthentication here rather
           than checking to see which supported auth method is required."""
        yield OAuthAuthentication(self.oauth_data, host, request_uri, headers, 
                response, content, self, self.__scheme)

def row_is_deleted(row):
    """Test if a row is marked as deleted.  Smart views 'maps' should not
    return rows that are marked as deleted, so this function is not often
    required."""
    try:
        return row['application_annotations']['Ubuntu One']\
                ['private_application_annotations']['deleted']
    except KeyError:
        return False


class CouchDatabaseBase(object):
    """An small records specific abstraction over a couch db database."""

    def __init__(self, database, uri, record_factory=None, create=False,
                 server_class=Server, **server_class_extras):
        self.server_uri = uri
        self._server = server_class(self.server_uri, **server_class_extras)
        if database not in self._server:
            if create:
                self._server.create(database)
            else:
                raise NoSuchDatabase(database)
        self.db = self._server[database]
        self.record_factory = record_factory or Record

    def _temporary_query(self, map_fun, reduce_fun=None, language='javascript',
            wrapper=None, **options):
        """Pass-through to CouchDB library.  Deprecated."""
        return self.db.query(map_fun, reduce_fun, language,
              wrapper, **options)

    def get_record(self, record_id):
        """Get a record from back end storage."""
        try:
            couch_record = self.db[record_id]
        except ResourceNotFound:
            return None
        data = {}
        if 'deleted' in couch_record.get('application_annotations', {}).get(
            'Ubuntu One', {}).get('private_application_annotations', {}):
            return None
        data.update(couch_record)
        record = self.record_factory(data=data)
        record.record_id = record_id
        return record

    def put_record(self, record):
        """Put a record in back end storage."""
        record_id = record.record_id or record._data.get('_id', '')
        record_data = record._data
        if record_id:
            self.db[record_id] = record_data
        else:
            record_id = self._add_record(record_data)
        return record_id

    def update_fields(self, record_id, fields):
        """Safely update a number of fields. 'fields' being a
        dictionary with fieldname: value for only the fields we want
        to change the value of.
        """
        while True:
            record = self.db[record_id]
            record.update(fields)
            try:
                self.db[record_id] = record
            except ResourceConflict:
                continue
            break

    def _add_record(self, data):
        """Add a new record to the storage backend."""
        return self.db.create(data)

    def delete_record(self, record_id):
        """Delete record with given id"""
        record = self.db[record_id]
        record.setdefault('application_annotations', {}).setdefault(
            'Ubuntu One', {}).setdefault('private_application_annotations', {})[
            'deleted'] = True
        self.db[record_id] = record

    def record_exists(self, record_id):
        """Check if record with given id exists."""
        if record_id not in self.db:
            return False
        record = self.db[record_id]
        return 'deleted' not in record.get('application_annotations', {}).get(
            'Ubuntu One', {}).get('private_application_annotations', {})

    def delete_view(self, view_name, design_doc=DEFAULT_DESIGN_DOCUMENT):
        """Remove a view, given its name.  Raises a KeyError on a unknown
        view.  Returns a dict of functions the deleted view defined."""
        if design_doc is None:
            design_doc = view_name

        doc_id = "_design/%(design_doc)s" % locals()

        # No atomic updates.  Only read & mutate & write.  Le sigh.
        # First, get current contents.
        try:
            view_container = self.db[doc_id]["views"]
        except (KeyError, ResourceNotFound):
            raise KeyError

        deleted_data = view_container.pop(view_name)  # Remove target

        if len(view_container) > 0:
            # Construct a new list of objects representing all views to have.
            views = [
                    ViewDefinition(design_doc, k, v.get("map"), v.get("reduce"))
                    for k, v
                    in view_container.iteritems()
                    ]
            # Push back a new batch of view.  Pray to Eris that this doesn't
            # clobber anything we want.

            # sync_many does nothing if we pass an empty list.  It even gets
            # its design-document from the ViewDefinition items, and if there
            # are no items, then it has no idea of a design document to
            # update.  This is a serious flaw.  Thus, the "else" to follow.
            ViewDefinition.sync_many(self.db, views, remove_missing=True)
        else:
            # There are no views left in this design document.

            # Remove design document.  This assumes there are only views in
            # design documents.  :(
            del self.db[doc_id]

        assert not self.view_exists(view_name, design_doc)

        return deleted_data

    def execute_view(self, view_name, design_doc=DEFAULT_DESIGN_DOCUMENT):
        """Execute view and return results."""
        if design_doc is None:
            design_doc = view_name

        view_id_fmt = "_design/%(design_doc)s/_view/%(view_name)s"
        return self.db.view(view_id_fmt % locals())

    def add_view(self, view_name, map_js, reduce_js,
            design_doc=DEFAULT_DESIGN_DOCUMENT):
        """Create a view, given a name and the two parts (map and reduce).
        Return the document id."""
        if design_doc is None:
            design_doc = view_name

        view = ViewDefinition(design_doc, view_name, map_js, reduce_js)
        view.sync(self.db)
        assert self.view_exists(view_name, design_doc)

    def view_exists(self, view_name, design_doc=DEFAULT_DESIGN_DOCUMENT):
        """Does a view with a given name, in a optional design document
        exist?"""
        if design_doc is None:
            design_doc = view_name

        doc_id = "_design/%(design_doc)s" % locals()

        try:
            view_container = self.db[doc_id]["views"]
            return view_name in view_container
        except (KeyError, ResourceNotFound):
            return False

    def list_views(self, design_doc):
        """Return a list of view names for a given design document.  There is
        no error if the design document does not exist or if there are no views
        in it."""
        doc_id = "_design/%(design_doc)s" % locals()
        try:
            return list(self.db[doc_id]["views"])
        except (KeyError, ResourceNotFound):
            return []

    def get_records(self, record_type=None, create_view=False,
            design_doc=DEFAULT_DESIGN_DOCUMENT):
        """A convenience function to get records from a view named
        C{get_records_and_type}.  We optionally create a view in the design
        document.  C{create_view} may be True or False, and a special value,
        None, is analogous to  O_EXCL|O_CREAT .

        Set record_type to a string to retrieve records of only that
        specified type. Otherwise, usse the view to return *all* records.
        If there is no view to use or we insist on creating a new view
        and cannot, raise KeyError .

        You can use index notation on the result to get rows with a
        particular record type.
        =>> results = get_records()
        =>> for foo_document in results["foo"]:
        ...    print foo_document

        Use slice notation to apply start-key and end-key options to the view.
        =>> results = get_records()
        =>> people = results[['Person']:['Person','ZZZZ']]
        """
        view_name = "get_records_and_type"
        view_map_js = """
            function(doc) {
                try {
                    if (! doc['application_annotations']['Ubuntu One']
                            ['private_application_annotations']['deleted']) {
                        emit(doc.record_type, doc);
                    }
                } catch (e) {
                    emit(doc.record_type, doc);
                }
            }"""

        if design_doc is None:
            design_doc = view_name

        exists = self.view_exists(view_name, design_doc)

        if exists:
            if create_view is None:
                raise KeyError("Exclusive creation failed.")
        else:
            if create_view == False:
                raise KeyError("View doesn't already exist.")

        if not exists:
            self.add_view(view_name, view_map_js, None, design_doc)

        viewdata = self.execute_view(view_name, design_doc)
        if record_type is None:
            return viewdata
        else:
            return viewdata[record_type]
