/* draw.cpp
 * loop that draws all the items in the presently-displayed part of
 * the score
 *  
 * for Denemo, a gtk+ frontend to GNU Lilypond
 * (c) 1999, 2000, 2001, 2002  Matthew Hiller, Adam Tee
 */

#include <stdio.h>
#include <string.h>
#include "calculatepositions.h"
#include "commandfuncs.h"
#include "contexts.h"
#include "draw.h"		/* Which includes gtk.h */
#include "drawingprims.h"
#include "gcs.h"
#include "slurs.h"
#include "hairpin.h"
#include "staffops.h"
#include "utils.h"
#include "pixmaps/toomany.xbm"
#include "exportmudela.h"	/* to generate lily text for display */
#include "lyparserfuncs.h"	/* to generate lily text for display */

/**
 * defines function to choose the correct FONT 
 * depending upon the GTK version
 *
 */
#if GTK_MAJOR_VERSION > 1
#define FONT (gtk_style_get_font(itp->widget->style))
#else
#define FONT (itp->widget->style->font)
#endif
#define REDEXCL_WIDTH 3
#define REDEXCL_HEIGHT 13

/**
 * scorearea_configure_event
 *
 * This function just creates a backing pixmap of the appropriate
 * size, recaculates the number of measures that can be fit into
 * the display, and returns 
 */

gint
scorearea_configure_event (GtkWidget * widget, GdkEventConfigure * event,
			   struct scoreinfo *si)
{
  /* Create a new backing pixmap of the appropriate size */
  if (si->pixmap)
    gdk_pixmap_unref (si->pixmap);

  si->pixmap = gdk_pixmap_new (widget->window,
			       widget->allocation.width,
			       widget->allocation.height, -1);

  set_width_to_work_with (si);
  nudgerightward (si);
  nudge_downward (si);
  return TRUE;
}

#define EXTRAFORSELECTRECT 2

/**
 *   Information to pass between the drawing functions
 */
struct infotopass
{
  gint clef;
  gint key;
  gint curaccs[7];
  gint keyaccs[7];
  gint stem_directive;
  gint time1, time2;
  gint tickspermeasure;
  gint wholenotewidth;
  gint objnum;
  gint measurenum;
  gint staffnum;
  gint top_y;
  gint y;
  gint markx1, markx2;
  gint marky1, marky2;
  /*GString *dynamic; */
  GtkWidget *widget;
  measurenode *curmeasure;
  GList *mwidthiterator;
  GSList *slur_stack;
  GSList *hairpin_stack;
};

/**
 *  draw_object
 *
 * Draws a single object in a measure within a staff. Return value: whether the
 * mudelaobject is off the end of the measure or not 
 */

static gboolean
draw_object (objnode * curobj, gint x, gint y,
	     struct scoreinfo *si, struct infotopass *itp)
{
  static GdkGC *blackgc = NULL;
  static GdkGC *redgc = NULL;
  //static GdkGC *greengc = NULL;
  mudelaobject *mudelaitem = (mudelaobject *) curobj->data;
  /* The current note, rest, etc. being painted */
  gint extra;

  if (!blackgc)
    blackgc = gcs_blackgc ();

  if (!redgc)
    redgc = gcs_redgc ();
  /* Should we set cursor-context info before drawing? */

  if (si->currentobject == curobj)
    {
      si->cursorclef = itp->clef;
      if (!si->cursor_appending)
	memcpy (si->cursoraccs, itp->curaccs, SEVENGINTS);
    }

  /* Now actually draw it */
  {
    GdkColor thecolor;
    const char *color = (mudelaitem->isinvisible) ? "white" : "black";
    gdk_color_parse (color, &thecolor);
    gdk_colormap_alloc_color (gdk_colormap_get_system (), &thecolor, TRUE,
			      TRUE);
    gdk_gc_set_foreground (blackgc, &thecolor);
  }

  switch (mudelaitem->type)
    {
    case CHORD:
      if (((chord *) mudelaitem->object)->is_figure)
	draw_figure (si->pixmap, blackgc, FONT,
		     x + mudelaitem->x, y, mudelaitem);
      else
	{
	  draw_chord (si->pixmap, blackgc, curobj, x + mudelaitem->x, y,
		      GPOINTER_TO_INT (itp->mwidthiterator->data),
		      itp->curaccs);
	}

      if (((chord *) mudelaitem->object)->lyric)
	draw_lyric (si->pixmap, blackgc, FONT,
		    x + mudelaitem->x, y, mudelaitem);

      if (((chord *) mudelaitem->object)->dynamics)
	draw_dynamic (si->pixmap, blackgc, FONT,
		      x + mudelaitem->x, y, mudelaitem);

      if (((chord *) mudelaitem->object)->slur_end_p)
	draw_slur (si->pixmap, blackgc, &(itp->slur_stack), x + mudelaitem->x,
		   y);
      if (((chord *) mudelaitem->object)->slur_begin_p)
	itp->slur_stack =
	  push_slur_stack (itp->slur_stack, x + mudelaitem->x);

      if (((chord *) mudelaitem->object)->crescendo_begin_p)
	itp->hairpin_stack =
	  push_hairpin_stack (itp->hairpin_stack, x + mudelaitem->x);
      else if (((chord *) mudelaitem->object)->diminuendo_begin_p)
	itp->hairpin_stack =
	  push_hairpin_stack (itp->hairpin_stack, x + mudelaitem->x);
      if (((chord *) mudelaitem->object)->crescendo_end_p)
	draw_hairpin (si->pixmap, blackgc, &(itp->hairpin_stack),
		      x + mudelaitem->x, y, 1);
      else if (((chord *) mudelaitem->object)->diminuendo_end_p)
	draw_hairpin (si->pixmap, blackgc, &(itp->hairpin_stack),
		      x + mudelaitem->x, y, 0);


      break;
    case TUPOPEN:
    case TUPCLOSE:
      draw_tupbracket (si->pixmap, blackgc, FONT,
		       x + mudelaitem->x, y, mudelaitem);
      break;
    case CLEF:
      draw_clef (si->pixmap, blackgc, x + mudelaitem->x, y,
		 itp->clef = ((clef *) mudelaitem->object)->type);
      if (si->currentobject == curobj && si->cursor_appending)
	si->cursorclef = itp->clef;
      break;
    case KEYSIG:
      draw_key (si->pixmap, blackgc, x + mudelaitem->x, y,
		((keysig *) mudelaitem->object)->number, itp->key,
		itp->clef, TRUE);
      itp->key = ((keysig *) mudelaitem->object)->number;
      memcpy (itp->keyaccs, ((keysig *) mudelaitem->object)->accs,
	      SEVENGINTS);
      memcpy (itp->curaccs, itp->keyaccs, SEVENGINTS);
      if (si->currentmeasure == itp->curmeasure)
	/* We're in the current measure */
	memcpy (si->nextmeasureaccs, itp->keyaccs, SEVENGINTS);
      break;
    case TIMESIG:
      draw_timesig (si->pixmap, blackgc, FONT,
		    x + mudelaitem->x, y,
		    itp->time1 = ((timesig *) mudelaitem->object)->time1,
		    itp->time2 = ((timesig *) mudelaitem->object)->time2);
      if (si->currentmeasure == itp->curmeasure)
	{
	  /* This is the current measure */
	  si->cursortime1 = itp->time1;
	  si->cursortime2 = itp->time2;
	}
      /* The following assumes no multiple simultaneous time signatures */
      itp->tickspermeasure = WHOLE_NUMTICKS * itp->time1 / itp->time2;
      break;
    case STEMDIRECTIVE:
      draw_stem_directive (si->pixmap, blackgc, FONT,
			   x + mudelaitem->x, y, mudelaitem);
      itp->stem_directive = ((stemdirective *) mudelaitem->object)->type;
      break;
    case LILYDIRECTIVE:
      draw_lily_dir (si->pixmap, redgc, FONT,
		     x + mudelaitem->x, y, mudelaitem);
      break;
      /*case DYNAMIC:
         draw_dynamic (si->pixmap, blackgc, FONT,
         x + mudelaitem->x, y, mudelaitem);
         itp->dynamic = mudelaitem->u.dynval.type; 
         break; */
    case GRACE_START:
    case GRACE_END:
      draw_gracebracket (si->pixmap, blackgc, FONT,
			 x + mudelaitem->x, y, mudelaitem);
      break;
    case BARLINE:
      drawbarline(si->pixmap, blackgc, x + mudelaitem->x, itp->top_y ,itp->y,
		  ((barline *) mudelaitem->object)->type);
      //g_print("Draw barline\n");
      break;
    default:
      /* Nothing */
      break;
    }
  if (si->currentobject == curobj)
    {				/* Draw the cursor */
      /* Determine if it needs to be red or not */
      if (si->cursor_appending || mudelaitem->type != CHORD)
	si->cursoroffend =
	  (mudelaitem->starttickofnextnote >= itp->tickspermeasure);
      else
	si->cursoroffend =
	  (mudelaitem->starttickofnextnote > itp->tickspermeasure);
      if (si->cursor_appending)
	{
	  extra = MAX (mudelaitem->minpixelsalloted,
		       space_after (mudelaitem->durinticks,
				    itp->wholenotewidth));
	  draw_cursor (si->pixmap, si, x + mudelaitem->x + extra, y,
		       si->cursorclef);
	  memcpy (si->cursoraccs, itp->curaccs, SEVENGINTS);
	}
      else
	draw_cursor (si->pixmap, si, x + mudelaitem->x, y, si->cursorclef);
      if (si->lily_file)
	{
	  if (mudelaitem->user_string == NULL)
	    {
	      mudelaitem->user_string = generate_lily (curobj);
	    }
	  display_string (mudelaitem->user_string, si);
	  /*      printf("The object at the cursor is <<%s>>\n", mudelaitem->user_string); */
	}
    }				/* End cursor drawing */


  /* Now quite possibly update the mark */

  if (si->firststaffmarked == itp->staffnum
      && si->firstmeasuremarked == itp->measurenum
      && si->firstobjmarked == itp->objnum)
    itp->markx1 = x + mudelaitem->x - EXTRAFORSELECTRECT;
  if (si->laststaffmarked == itp->staffnum
      && si->lastmeasuremarked == itp->measurenum
      && si->lastobjmarked == itp->objnum)
    itp->markx2 = x + mudelaitem->x + mudelaitem->minpixelsalloted
      + EXTRAFORSELECTRECT;

  /* And give a return value and we're done */
  if (mudelaitem->isinvisible)
    {
      GdkColor thecolor;
      gdk_color_parse ("black", &thecolor);
      gdk_colormap_alloc_color (gdk_colormap_get_system (), &thecolor, TRUE,
				TRUE);
      gdk_gc_set_foreground (blackgc, &thecolor);
    }

  return (mudelaitem->starttickofnextnote > itp->tickspermeasure);
}

/* Draws a single measure within a staff. */

void
draw_measure (measurenode * curmeasure, gint x, gint y,
	      struct scoreinfo *si, struct infotopass *itp)
{
  static GdkPixmap *redexclaim = NULL;
  static GdkGC *redgc;
  static GdkGC *blackgc;
  static GdkFont *mnumfont;
  static GString *mstring;
  gboolean offend = FALSE;
  objnode *curobj;

  /* initialization */
  if (!redexclaim)
    {
      redexclaim = bitmaphelper (itp->widget, toomany);
      mnumfont =
	gdk_fontset_load
	("-*-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-*");
      if (!mnumfont)
	mnumfont = FONT;
      mstring = g_string_new (NULL);
      redgc = gcs_redgc ();
      blackgc = gcs_blackgc ();
    }

  /* Set information about the state at the current measure, if necessary */

  memcpy (itp->curaccs, itp->keyaccs, SEVENGINTS);
  itp->wholenotewidth = si->measurewidth * itp->time2 / itp->time1;
  if (curmeasure == si->currentmeasure)
    {
      si->curmeasureclef = itp->clef;
      memcpy (si->curmeasureaccs, itp->keyaccs, SEVENGINTS);
      memcpy (si->nextmeasureaccs, itp->keyaccs, SEVENGINTS);
      si->curmeasurekey = itp->key;
      si->curmeasure_stem_directive = itp->stem_directive;
      si->cursortime1 = itp->time1;
      si->cursortime2 = itp->time2;
    }

  /* If this is the current staff, paint the measure number at the
   * preceding barline */

  if (si->currentstaffnum == itp->staffnum)
    {
      g_string_sprintf (mstring, "%d", itp->measurenum);
      gdk_draw_text (si->pixmap, mnumfont, blackgc, x - SPACE_FOR_BARLINE,
		     y - 2, mstring->str, mstring->len);
      if (si->currentmeasurenum == itp->measurenum && !si->currentobject)
	{
	  /* That is, the cursor's at the beginning of this blank measure */
	  si->cursoroffend = FALSE;
	  draw_cursor (si->pixmap, si, x, y, itp->clef);
	  memcpy (si->cursoraccs, itp->curaccs, SEVENGINTS);
	  si->cursorclef = itp->clef;
	}
    }
  curobj = (objnode *) curmeasure->data;
  /* These default values for the markx'es may be necessary down
   * the road */
  if (si->firststaffmarked == itp->staffnum
      && si->firstmeasuremarked == itp->measurenum)
    {
      if (!curobj)
	itp->markx1 = x - EXTRAFORSELECTRECT;
      else
	itp->markx1 = x + GPOINTER_TO_INT (itp->mwidthiterator->data);
    }
  if (si->laststaffmarked == itp->staffnum
      && si->lastmeasuremarked == itp->measurenum)
    {
      if (!curobj
	  || (si->lastobjmarked >= gint (g_list_length ((objnode *) curobj))))
	itp->markx2 =
	  x + GPOINTER_TO_INT (itp->mwidthiterator->data) +
	  SPACE_FOR_BARLINE + EXTRAFORSELECTRECT;
      else
	itp->markx2 = x;
    }
  /* Paint the red exclamation point, if necessary */

  /* Draw each mudelaitem */
  for (itp->objnum = 0; curobj; curobj = curobj->next, itp->objnum++)
    offend = draw_object (curobj, x, y, si, itp);

  if (offend)
    drawbitmapinverse (si->pixmap, redgc, redexclaim,
		       x, y - 8 - REDEXCL_HEIGHT,
		       REDEXCL_WIDTH, REDEXCL_HEIGHT);
}

/* Draws a single staff */

void
draw_staff (staff * curstaffstruct, gint y,
	    struct scoreinfo *si, struct infotopass *itp)
{
  static GdkFont *staffnamefont = NULL;
  static GdkGC *blackgc;
  gint x, i;
  

  if (!staffnamefont)
    {
      staffnamefont =
	gdk_fontset_load
	("-*-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-*");
      if (!staffnamefont)
	staffnamefont = FONT;
      blackgc = gcs_blackgc ();
    }
  draw_clef (si->pixmap, blackgc, LEFT_MARGIN, y,
	     itp->clef = curstaffstruct->leftmost_clefcontext);
  x = KEY_MARGIN;
  draw_key (si->pixmap, blackgc, x, y,
	    itp->key = curstaffstruct->leftmost_keysigcontext,
	    0, itp->clef, TRUE);
  memcpy (itp->keyaccs, curstaffstruct->leftmost_keyaccs, SEVENGINTS);
  x += si->maxkeywidth;
  draw_timesig (si->pixmap, blackgc, FONT, x, y,
		itp->time1 = curstaffstruct->leftmost_time1context,
		itp->time2 = curstaffstruct->leftmost_time2context);
  x += SPACE_FOR_TIME;
  itp->stem_directive = curstaffstruct->leftmost_stem_directive;
  itp->tickspermeasure = WHOLE_NUMTICKS * itp->time1 / itp->time2;
  if (si->firststaffmarked == itp->staffnum)
    itp->marky1 = y - EXTRAFORSELECTRECT;
  if (si->laststaffmarked == itp->staffnum)
    itp->marky2 = y + STAFF_HEIGHT + EXTRAFORSELECTRECT;

  gint buffer = (curstaffstruct->voicenumber == 1) ? 8 : 
    (curstaffstruct->voicenumber == 2) ? 0 : -8;
  //g_print("Buffer %d, ", buffer);
    /* Draw staff name */
    gdk_draw_text (si->pixmap, staffnamefont, blackgc, KEY_MARGIN, y - buffer,
		   curstaffstruct->denemo_name->str,
		   curstaffstruct->denemo_name->len);
 

  /* Loop that will draw each measure. Basically a for loop, but was uglier
   * when written that way.  */
  itp->measurenum = si->leftmeasurenum;
  itp->curmeasure =
    g_list_nth (curstaffstruct->measures, itp->measurenum - 1);
  itp->mwidthiterator = g_list_nth (si->measurewidths, itp->measurenum - 1);

  while (itp->measurenum <= si->rightmeasurenum)
    {
      draw_measure (itp->curmeasure, x, y, si, itp);
      x += GPOINTER_TO_INT (itp->mwidthiterator->data) + SPACE_FOR_BARLINE;
      itp->curmeasure = itp->curmeasure->next;
      itp->mwidthiterator = itp->mwidthiterator->next;
      itp->measurenum++;
    }

  /* now draw the staff lines, reset itp->slur_stack, and we're done */
  for (i = 0; i < curstaffstruct->no_of_lines; i++, y += LINE_SPACE)
    gdk_draw_line (si->pixmap, blackgc, LEFT_MARGIN, y,
		   x - HALF_BARLINE_SPACE, y);
  /* Initialize the slur_stack for this staff. For the time being,
     slurs that begin and/or end after the portion of the music
     that is shown are not drawn. */
  if (itp->slur_stack)
    {
      g_slist_free (itp->slur_stack);
      itp->slur_stack = NULL;
    }
}

/* This actually draws the score, staff-by-staff */

void
draw_score (GtkWidget * widget, struct scoreinfo *si)
{
  staffnode *curstaff;
  gint x, y, i;
  struct infotopass itp;
  static GdkGC *blackgc = NULL;
  static GdkGC *bluegc;
  GList *mwidthiterator;

  /* One-time inits */
  if (!blackgc)
    {
      blackgc = gcs_blackgc ();
      bluegc = gcs_bluegc ();
    }

  /* Initialize some fields in itp */

  itp.widget = widget;
  itp.slur_stack = NULL;

  /* Draw each staff */
  for ((itp.staffnum = si->top_staff,
	curstaff = g_list_nth (si->thescore, si->top_staff - 1),
	y = si->staffspace / 4);
       curstaff && itp.staffnum <= si->bottom_staff; itp.staffnum++)
    {
      y += ((staff *) curstaff->data)->space_above;
      
      gint top_y = (si->staffspace / 4) +
	((staff *) curstaff->data)->space_above;
      //      g_print("top_y %d", top_y);
      itp.top_y = top_y;
      if (curstaff && ((staff *) curstaff->data)->voicenumber == 1
	  && itp.staffnum < si->bottom_staff)
	itp.y = y+(si->staffspace + ((staff *) curstaff->data)->space_below);
      else
	itp.y=y;
      draw_staff ((staff *) curstaff->data, y, si, &itp);
      curstaff = curstaff->next;
      if (curstaff && ((staff *) curstaff->data)->voicenumber == 1
	  && itp.staffnum < si->bottom_staff)
	y += (si->staffspace + ((staff *) curstaff->data)->space_below);
    }

  /* Now draw the barlines between the measures, across all the staffs */
  curstaff = g_list_nth (si->thescore, si->top_staff - 1);
  y += STAFF_HEIGHT;
  mwidthiterator = g_list_nth (si->measurewidths, si->leftmeasurenum - 1);
  for (x = KEY_MARGIN + si->maxkeywidth + SPACE_FOR_TIME - HALF_BARLINE_SPACE,
       i = si->leftmeasurenum;
       i <= si->rightmeasurenum; mwidthiterator = mwidthiterator->next, i++)
    {
      gint top_y = (si->staffspace / 4) +
	((staff *) curstaff->data)->space_above;
      x += GPOINTER_TO_INT (mwidthiterator->data) + SPACE_FOR_BARLINE;
      if (!mwidthiterator->next)	/* Last measure - draw double-barline */
	x -= 3;
      gdk_draw_line (si->pixmap, blackgc, x, top_y, x, y);

      if (!mwidthiterator->next)
	{
	  /* Again, we've reached the end of the score and should
	   * draw a double-barline */
	  x += 3;
	  gdk_draw_rectangle (si->pixmap, blackgc, TRUE, x,
			      top_y, 4,
			      y - si->staffspace / 4 + 1 -
			      ((staff *) curstaff->data)->space_above);
	}
    }				/* End barline drawing loop */
  /* Draw the selection rectangle */
  if (si->markstaffnum)
    draw_selection (si->pixmap, bluegc, itp.markx1, itp.marky1, itp.markx2,
		    itp.marky2);
  /* And we're done */
}

/* Here we have the function that actually draws the score. Note that
 * it does not clip intelligently at all */

gint
scorearea_expose_event (GtkWidget * widget, GdkEventExpose * event,
			gpointer data)
{
  struct scoreinfo *si = (struct scoreinfo *) data;

  /* Clear the backing pixmap */

  if (GTK_WIDGET_IS_SENSITIVE (si->scorearea))
    gdk_draw_rectangle (si->pixmap,
			widget->style->white_gc,
			TRUE,
			0, 0,
			widget->allocation.width, widget->allocation.height);
  else
    gdk_draw_rectangle (si->pixmap,
			widget->style->bg_gc[0],
			TRUE,
			0, 0,
			widget->allocation.width, widget->allocation.height);

  /* Draw the score */
  draw_score (widget, si);

  /* Now actually draw the backing pixmap onto the drawing area */

  gdk_draw_pixmap (si->scorearea->window,
		   si->scorearea->style->black_gc,
		   si->pixmap,
		   0, 0, 0, 0,
		   si->scorearea->allocation.width,
		   si->scorearea->allocation.height);

  return TRUE;
}

#undef FONT


/* 
 * create an editable display of the passed str at the bottom 
 * of the drawing area 
 */
void
display_string (gchar * str, struct scoreinfo *si)
{
#if GTK_MAJOR_VERSION > 1
  if (GTK_WIDGET_VISIBLE (si->scorearea))	/* not in toplevel */
    set_text_node (NULL, si);
#endif

}
