/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by:
 *               Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 */

/**
 * SECTION:dee-sequence-model
 * @short_description: A #DeeModel<!-- --> implementation backed by a #GSequence
 * @include: dee.h
 *
 * #DeeSequenceModel is an implementation of the #DeeModel<!-- --> interface
 * backed by a #GSequence. It extends #DeeVersionedModel sp that you may use
 * it as back end model for a #DeeShareedModel.
 *
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <memory.h>
#include <time.h>
#include <unistd.h>
#include <gobject/gvaluecollector.h>

#include "dee-model.h"
#include "dee-versioned-model.h"
#include "dee-sequence-model.h"
#include "dee-marshal.h"
#include "trace-log.h"

static void dee_sequence_model_model_iface_init (DeeModelIface *iface);

G_DEFINE_TYPE_WITH_CODE (DeeSequenceModel,
                         dee_sequence_model,
                         DEE_TYPE_VERSIONED_MODEL,
                         G_IMPLEMENT_INTERFACE (DEE_TYPE_MODEL,
                                                dee_sequence_model_model_iface_init));

#define DEE_SEQUENCE_MODEL_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE(obj, DEE_TYPE_SEQUENCE_MODEL, DeeSequenceModelPrivate))

/**
 * DeeSequenceModelPrivate:
 *
 * Ignore this structure.
 */
struct _DeeSequenceModelPrivate
{
  /* Row data stored in GValueArrays */
  GSequence *sequence;

  /* Flag marking if we are in a transaction */
  gboolean   setting_many;
  
  /* Signal freeze count */
  guint      freeze_count;
  
  /* List of gpointer[2] with [SignalType,*iter] */
  GSList    *frozen_signals;
};

typedef enum 
{
  SIGNAL_TYPE_ROW_ADDED,
  SIGNAL_TYPE_ROW_CHANGED,
} SignalType;

/*
 * DeeModel forward declarations
 */
static guint          dee_sequence_model_get_n_rows     (DeeModel *self);

static DeeModelIter*  dee_sequence_model_append_valist  (DeeModel *self,
                                                         va_list    args);

static DeeModelIter*  dee_sequence_model_prepend_valist  (DeeModel *self,
                                                          va_list    args);

static DeeModelIter*  dee_sequence_model_insert_before_valist (DeeModel     *self,
                                                              DeeModelIter *iter,
                                                              va_list    args);

static void           dee_sequence_model_remove         (DeeModel     *self,
                                                          DeeModelIter *iter);

static void           dee_sequence_model_set_valist     (DeeModel       *self,
                                                          DeeModelIter   *iter,
                                                          va_list          args);

static void           dee_sequence_model_set_value      (DeeModel       *self,
                                                         DeeModelIter   *iter,
                                                         guint            column,
                                                         const GValue    *value);

static void           dee_sequence_model_set_value_silently (DeeModel       *self,
                                                             DeeModelIter   *iter,
                                                             guint            column,
                                                             const GValue    *value);


static void           dee_sequence_model_get_value      (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column,
                                                         GValue        *value);

static const gchar*  dee_sequence_model_get_string      (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint         column);

static DeeModelIter* dee_sequence_model_get_first_iter  (DeeModel     *self);

static DeeModelIter* dee_sequence_model_get_last_iter   (DeeModel     *self);

static DeeModelIter* dee_sequence_model_get_iter_at_row (DeeModel     *self,
                                                         guint          row);

static DeeModelIter* dee_sequence_model_next            (DeeModel     *self,
                                                         DeeModelIter *iter);

static DeeModelIter* dee_sequence_model_prev            (DeeModel     *self,
                                                         DeeModelIter *iter);

static gboolean       dee_sequence_model_is_first       (DeeModel     *self,
                                                         DeeModelIter *iter);

static gboolean       dee_sequence_model_is_last        (DeeModel     *self,
                                                         DeeModelIter *iter);

static gint           dee_sequence_model_get_position   (DeeModel     *self,
                                                         DeeModelIter *iter);

static void           dee_sequence_model_freeze_signals (DeeModel     *self);

static void           dee_sequence_model_thaw_signals   (DeeModel     *self);

/*
 * Private forwards
 */
static GValueArray*   dee_sequence_model_create_empty_row (DeeModel *self);

static void           dee_sequence_model_emit_signal (DeeModel     *self,
                                                      SignalType    signal_type,
                                                      DeeModelIter *iter);

static void           dee_sequence_model_check_signal (DeeModel     *self,
                                                       SignalType    signal_type,
                                                       DeeModelIter *iter);

/* GObject Init */
static void
dee_sequence_model_finalize (GObject *object)
{
  //DeeSequenceModelPrivate *priv = DEE_SEQUENCE_MODEL (object)->priv;

  G_OBJECT_CLASS (dee_sequence_model_parent_class)->finalize (object);
}

static void
dee_sequence_model_set_property (GObject      *object,
                                 guint         id,
                                 const GValue *value,
                                 GParamSpec   *pspec)
{
  switch (id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dee_sequence_model_get_property (GObject    *object,
                                 guint       id,
                                 GValue     *value,
                                 GParamSpec *pspec)
{
  switch (id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dee_sequence_model_class_init (DeeSequenceModelClass *klass)
{
  GObjectClass  *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize     = dee_sequence_model_finalize;  
  obj_class->set_property = dee_sequence_model_set_property;
  obj_class->get_property = dee_sequence_model_get_property;

  /* Add private data */
  g_type_class_add_private (obj_class, sizeof (DeeSequenceModelPrivate));
}

static void
dee_sequence_model_model_iface_init (DeeModelIface *iface)
{
  iface->get_n_rows           = dee_sequence_model_get_n_rows;
  iface->prepend_valist       = dee_sequence_model_prepend_valist;
  iface->append_valist        = dee_sequence_model_append_valist;
  iface->insert_before_valist = dee_sequence_model_insert_before_valist;
  iface->remove               = dee_sequence_model_remove;
  iface->set_valist           = dee_sequence_model_set_valist;
  iface->set_value            = dee_sequence_model_set_value;
  iface->set_value_silently   = dee_sequence_model_set_value_silently;
  iface->get_value            = dee_sequence_model_get_value;
  iface->get_string           = dee_sequence_model_get_string;
  iface->get_first_iter       = dee_sequence_model_get_first_iter;
  iface->get_last_iter        = dee_sequence_model_get_last_iter;
  iface->get_iter_at_row      = dee_sequence_model_get_iter_at_row;
  iface->next                 = dee_sequence_model_next;
  iface->prev                 = dee_sequence_model_prev;
  iface->is_first             = dee_sequence_model_is_first;
  iface->is_last              = dee_sequence_model_is_last;
  iface->get_position         = dee_sequence_model_get_position;
  iface->freeze_signals       = dee_sequence_model_freeze_signals;
  iface->thaw_signals         = dee_sequence_model_thaw_signals;
}

static void
dee_sequence_model_init (DeeSequenceModel *model)
{
  DeeSequenceModelPrivate *priv;

  priv = model->priv = DEE_SEQUENCE_MODEL_GET_PRIVATE (model);
  priv->sequence = g_sequence_new (NULL);
  priv->setting_many = FALSE;
  priv->freeze_count = 0;
  priv->frozen_signals = NULL;
}

/* Private Methods */

static void
dee_sequence_model_emit_signal (DeeModel     *self,
                                SignalType    signal_type,
                                DeeModelIter *iter)
{
  switch (signal_type)
    {
      case SIGNAL_TYPE_ROW_ADDED:
        g_signal_emit_by_name (self, "row-added", iter);
        break;
      case SIGNAL_TYPE_ROW_CHANGED:
        g_signal_emit_by_name (self, "row-changed", iter);
        break;
    }
    
}

static void
dee_sequence_model_check_signal (DeeModel     *self,
                                 SignalType    signal_type,
                                 DeeModelIter *iter)
{
  DeeSequenceModelPrivate *priv;
  
  priv = DEE_SEQUENCE_MODEL (self)->priv;
  
  if (priv->freeze_count == 0)
    dee_sequence_model_emit_signal (self, signal_type, iter);
  else
    {
      gpointer *signal_ctx = g_new (gpointer, 2);
      signal_ctx[0] = GUINT_TO_POINTER (signal_type);
      signal_ctx[1] = iter;
      priv->frozen_signals = g_slist_prepend (priv->frozen_signals, signal_ctx);
    }
}

/*
 * DeeModel Interface Implementation
 */

static guint
dee_sequence_model_get_n_rows (DeeModel *self)
{
  DeeSequenceModelPrivate *priv;

  g_return_val_if_fail (DEE_IS_SEQUENCE_MODEL (self), 0);
  priv = ((DeeSequenceModel *) self)->priv;

  return g_sequence_get_length (priv->sequence);
}

static DeeModelIter*
dee_sequence_model_prepend_valist (DeeModel *self,
                                   va_list    args)
{
  DeeSequenceModel        *_self = (DeeSequenceModel *) self;
  DeeSequenceModelPrivate *priv;
  DeeModelIter            *iter;
  GValueArray             *array;  

  g_return_val_if_fail (DEE_IS_SEQUENCE_MODEL (_self), NULL);

  priv = _self->priv;
  array = dee_sequence_model_create_empty_row (self);
  iter = (DeeModelIter*) g_sequence_prepend (priv->sequence, array);
  
  priv->setting_many = TRUE;
  dee_model_set_valist (self, iter, args);
  priv->setting_many = FALSE;

  dee_versioned_model_prepend_next_seqnum (self);
  dee_sequence_model_check_signal (self, SIGNAL_TYPE_ROW_ADDED, iter);
  
  return iter;
}

static DeeModelIter*
dee_sequence_model_append_valist (DeeModel *self,
                                  va_list    args)
{
  DeeSequenceModel        *_self = (DeeSequenceModel *) self;
  DeeSequenceModelPrivate *priv;
  DeeModelIter            *iter;
  GValueArray             *array;

  g_return_val_if_fail (DEE_IS_SEQUENCE_MODEL (_self), NULL);

  priv = _self->priv;
  array = dee_sequence_model_create_empty_row (self);
  iter = (DeeModelIter*) g_sequence_append (priv->sequence, array);
  
  priv->setting_many = TRUE;
  dee_model_set_valist (self, iter, args);
  priv->setting_many = FALSE;

  dee_versioned_model_append_next_seqnum (self);
  dee_sequence_model_check_signal (self, SIGNAL_TYPE_ROW_ADDED, iter);
  
  return iter;
}

static DeeModelIter*
dee_sequence_model_insert_before_valist (DeeModel *self,
                                         DeeModelIter *iter,
                                         va_list args)
{
  DeeSequenceModelPrivate *priv;
  GValueArray             *array;
  guint                    pos;

  g_return_val_if_fail (DEE_IS_SEQUENCE_MODEL (self), NULL);

  priv = DEE_SEQUENCE_MODEL (self)->priv;
  array = dee_sequence_model_create_empty_row (self);
  iter = (DeeModelIter*) g_sequence_insert_before ((GSequenceIter *) iter,
                                                   array);

  

  priv->setting_many = TRUE;
  dee_model_set_valist (self, iter, args);
  priv->setting_many = FALSE;

  pos = dee_model_get_position (self, iter);
  dee_versioned_model_insert_next_seqnum (self, pos);
  dee_sequence_model_check_signal (self, SIGNAL_TYPE_ROW_ADDED, iter);
  
  return iter;
}

static void
dee_sequence_model_remove (DeeModel     *self,
                           DeeModelIter *iter_)
{
  DeeSequenceModel        *_self = (DeeSequenceModel *)self;
  DeeSequenceModelPrivate *priv;
  GSequenceIter           *iter = (GSequenceIter *)iter_;

  g_return_if_fail (DEE_IS_SEQUENCE_MODEL (_self));
  priv = _self->priv;

  if (iter_ == dee_model_get_last_iter (self))
    return;

  if (iter)
    {
      GValueArray *array;
      gint         pos;      
      
      /* Emit the removed signal while the iter is still valid,
       * but after we increased the seqnum */
      pos = g_sequence_iter_get_position (iter);
      dee_versioned_model_inc_seqnum (self, pos);      
      
      g_signal_emit_by_name (self, "row-removed", iter);           

      array = g_sequence_get (iter);
      g_value_array_free (array);
      g_sequence_remove (iter);
      dee_versioned_model_remove_seqnum (self, pos);
    }
  else
    {
      g_warning ("Unable to remove row '%p': does not exists", iter);
    }
}

static void
dee_sequence_model_set_valist (DeeModel       *self,
                               DeeModelIter   *iter,
                               va_list          args)
{
  DeeSequenceModel        *_self = (DeeSequenceModel *) self;
  DeeSequenceModelPrivate *priv;
  guint                    column = 0;
  guint                    n_columns;
  gboolean                 was_in_transaction;

  g_return_if_fail (DEE_SEQUENCE_MODEL (_self));

  priv = _self->priv;
  n_columns = dee_model_get_n_columns ((DeeModel *) self);
  column = va_arg (args, gint);

  was_in_transaction = priv->setting_many;
  priv->setting_many = TRUE;
  while (column != -1)
    {
      GValue  value = { 0, };
      gchar  *error = NULL;

      if (column < 0 || column >= n_columns)
        {
          g_warning ("Column %d is out of range", column);
          break;
        }

      g_value_init (&value,
                    dee_model_get_column_type ((DeeModel *) self, column));

      G_VALUE_COLLECT (&value, args, 0, &error);
      if (error)
        {
          g_warning ("%s: %s", G_STRLOC, error);
          g_free (error);
          break;
        }

      dee_model_set_value (self, iter, column, &value);

      g_value_unset (&value);

      column = va_arg (args, gint);
    }
  priv->setting_many = was_in_transaction;

  /* If we did something column will be != 0 */
  if (column != 0 && !priv->setting_many)
    {
      guint pos = dee_model_get_position (self, iter);
      dee_versioned_model_inc_seqnum (self, pos);
      dee_sequence_model_check_signal (self, SIGNAL_TYPE_ROW_CHANGED, iter);
    }
}

static void
dee_sequence_model_set_value (DeeModel      *self,
                              DeeModelIter  *iter,
                              guint           column,
                              const GValue   *value)
{
  DeeSequenceModel *_self = (DeeSequenceModel *)self;
  DeeSequenceModelPrivate *priv;
  gint              pos;

  g_return_if_fail (DEE_IS_SEQUENCE_MODEL (_self));
  g_return_if_fail (iter != NULL);
  
  priv = _self->priv;
  
  dee_model_set_value_silently (self, iter, column, value);
  
  if (priv->setting_many == FALSE)
    {
      pos = g_sequence_iter_get_position ((GSequenceIter*)iter);
      dee_versioned_model_inc_seqnum (self, pos);
      
      dee_sequence_model_check_signal (self, SIGNAL_TYPE_ROW_CHANGED, iter);
    }
}

static void
dee_sequence_model_set_value_silently (DeeModel      *self,
                                       DeeModelIter  *iter,
                                       guint           column,
                                       const GValue   *value)
{
  DeeSequenceModel *_self = (DeeSequenceModel *)self;
  DeeSequenceModelPrivate *priv;
  GValueArray      *value_array;
  GValue           *iter_value;

  g_return_if_fail (DEE_IS_SEQUENCE_MODEL (_self));
  g_return_if_fail (iter != NULL);
  
  priv = _self->priv;

  value_array = g_sequence_get ((GSequenceIter *) iter);
  iter_value = g_value_array_get_nth (value_array, column);

  if (G_VALUE_TYPE (value) != G_VALUE_TYPE (iter_value))
    {
      g_critical ("Illegal internal state when setting value: "
                  "Column type mismatch");
      return;
    }

  g_value_copy (value, iter_value);
}

static void
dee_sequence_model_get_value (DeeModel     *self,
                              DeeModelIter *iter,
                              guint          column,
                              GValue        *value)
{
  DeeSequenceModel *_self = (DeeSequenceModel *)self;
  GValueArray      *value_array;
  GValue           *iter_value;

  g_return_if_fail (DEE_IS_SEQUENCE_MODEL (_self));
  g_return_if_fail (iter != NULL);
  g_return_if_fail (value != NULL);
  g_return_if_fail (column < dee_model_get_n_columns (self));

  g_value_init (value, dee_model_get_column_type (self, column));

  value_array = g_sequence_get ((GSequenceIter *) iter);
  iter_value = g_value_array_get_nth (value_array, column);

  /* Copy the value if it's a valid value type. We treat strings specially
   * to avoid copying them. All other valid value types are passed  */
  if (G_VALUE_TYPE (value) != G_VALUE_TYPE (iter_value))
    {
      g_critical ("Illegal internal state when getting value: Type mismatch");
      return;
    }
  else
    g_value_copy (iter_value, value);
}

static const gchar*
dee_sequence_model_get_string (DeeModel *self, DeeModelIter *iter, guint column)
{
  GValueArray      *value_array;
  GValue           *iter_value;

  g_return_val_if_fail (DEE_IS_SEQUENCE_MODEL (self), NULL);
  g_return_val_if_fail (iter != NULL, NULL);
  g_return_val_if_fail (column < dee_model_get_n_columns (self), NULL);

  value_array = g_sequence_get ((GSequenceIter *) iter);
  iter_value = g_value_array_get_nth (value_array, column);

  if (dee_model_get_column_type (self, column) != G_TYPE_STRING)
    {
      g_critical ("Column %u does not hold a string. Found %s",
                  column,
                  g_type_name (dee_model_get_column_type (self, column)));
      return NULL;
    }

  if (iter_value == NULL)
    {
      g_critical ("Internal error. No value at column %u", column);
      return NULL;
    }

  return g_value_get_string (iter_value);
}

static DeeModelIter*
dee_sequence_model_get_first_iter (DeeModel     *self)
{
  DeeSequenceModel *_self = (DeeSequenceModel *)self;
  
  g_return_val_if_fail (DEE_IS_SEQUENCE_MODEL (_self), NULL);

  return (DeeModelIter *) g_sequence_get_begin_iter (_self->priv->sequence);
}

static DeeModelIter*
dee_sequence_model_get_last_iter (DeeModel *self)
{
  DeeSequenceModel *_self = (DeeSequenceModel *)self;

  g_return_val_if_fail (DEE_IS_SEQUENCE_MODEL (_self), NULL);

  return (DeeModelIter *) g_sequence_get_end_iter (_self->priv->sequence);
}

static DeeModelIter*
dee_sequence_model_get_iter_at_row (DeeModel *self, guint row)
{
  DeeSequenceModel *_self = (DeeSequenceModel *)self;
  
  g_return_val_if_fail (DEE_IS_SEQUENCE_MODEL (self), NULL);

  return (DeeModelIter *) g_sequence_get_iter_at_pos (_self->priv->sequence,
                                                      row);
}

static DeeModelIter*
dee_sequence_model_next (DeeModel     *self,
                         DeeModelIter *iter)
{
  g_return_val_if_fail (DEE_IS_SEQUENCE_MODEL (self), NULL);
  g_return_val_if_fail (iter, NULL);
  g_return_val_if_fail (!dee_model_is_last (self, iter), NULL);

  return (DeeModelIter *) g_sequence_iter_next ((GSequenceIter *)iter);
}

static DeeModelIter*
dee_sequence_model_prev (DeeModel     *self,
                         DeeModelIter *iter)
{
  g_return_val_if_fail (DEE_IS_SEQUENCE_MODEL (self), NULL);
  g_return_val_if_fail (iter, NULL);
  g_return_val_if_fail (!dee_model_is_first (self, iter), NULL);

  return (DeeModelIter *) g_sequence_iter_prev ((GSequenceIter *)iter);
}

static gboolean
dee_sequence_model_is_first (DeeModel     *self,
                             DeeModelIter *iter)
{
  g_return_val_if_fail (DEE_IS_SEQUENCE_MODEL (self), FALSE);
  g_return_val_if_fail (iter, FALSE);

  return g_sequence_iter_is_begin ((GSequenceIter *)iter);
}

static gboolean
dee_sequence_model_is_last (DeeModel     *self,
                            DeeModelIter *iter)
{
  g_return_val_if_fail (DEE_IS_SEQUENCE_MODEL (self), FALSE);
  g_return_val_if_fail (iter, FALSE);

  return g_sequence_iter_is_end ((GSequenceIter *)iter);
}

static gint
dee_sequence_model_get_position (DeeModel     *self,
                                 DeeModelIter *iter)
{
  g_return_val_if_fail (DEE_IS_SEQUENCE_MODEL (self), FALSE);
  g_return_val_if_fail (iter, FALSE);

  return g_sequence_iter_get_position ((GSequenceIter *)iter);
}

static void
dee_sequence_model_freeze_signals (DeeModel     *self)
{
  g_return_if_fail (DEE_IS_SEQUENCE_MODEL (self));
  
  DEE_SEQUENCE_MODEL (self)->priv->freeze_count++;
}

static void
dee_sequence_model_thaw_signals (DeeModel     *self)
{
  DeeSequenceModelPrivate *priv;
  
  g_return_if_fail (DEE_IS_SEQUENCE_MODEL (self));
  
  priv =  DEE_SEQUENCE_MODEL (self)->priv;
  priv->freeze_count--;
  
  /* Replay signals if we are completely thawed */
  if (priv->freeze_count == 0)
    {
      /* We always prepend the signal contexts, so reverse the list first */
      priv->frozen_signals = g_slist_reverse (priv->frozen_signals);
    
      GSList   *iter = priv->frozen_signals;
      gpointer  *signal_ctx;
      
      for (; iter != NULL; iter = iter->next)
        {
          signal_ctx = (gpointer*) iter->data;
          dee_sequence_model_emit_signal (self,
                                          GPOINTER_TO_UINT (signal_ctx[0]),
                                          (DeeModelIter *) signal_ctx[1]);
          g_free (signal_ctx);
        }
      
      if (priv->frozen_signals)
        g_slist_free (priv->frozen_signals);
      
      priv->frozen_signals = NULL;
    }
}

/*
 * Private methods
 */
 /* Create a GValyeArray set up with the right GValue types for this model */
static GValueArray*
dee_sequence_model_create_empty_row (DeeModel *self)
{
  DeeSequenceModelPrivate *priv;
  GValueArray             *array;
  gint                     i;
  guint                    n_columns;

  priv = ((DeeSequenceModel *)self)->priv;
  n_columns = dee_model_get_n_columns (self);
  array = g_value_array_new (dee_model_get_n_columns (self) + 1);

  for (i = 0; i < n_columns; i++)
    {
      GValue *value = NULL;

      g_value_array_append (array, NULL);

      value = g_value_array_get_nth (array, i);
      g_value_init (value,
                    dee_model_get_column_type (self, i));
    }

  return array;
}

/*
 * Constructors
 */
DeeModel*
dee_sequence_model_new (guint n_columns, ...)
{
  DeeModel *self;
  va_list   args;
  gint      i;

  self = DEE_MODEL (g_object_new (DEE_TYPE_SEQUENCE_MODEL, NULL));

  dee_model_set_n_columns (self, n_columns);

  va_start (args, n_columns);

  for (i = 0; i < n_columns; i++)
    {
      GType type = va_arg (args, GType);

      if (!dee_model_check_type (type))
        {
          g_warning ("DeeSequenceModel: %s is an invalid type",
                     g_type_name (type));
          g_object_unref (self);
          return NULL;
        }

      dee_model_set_column_type (self, i, type);
    }

  va_end (args);

  return self;
}