/*
 * This is a helper library to ease testing of D-Bus objects.
 * Copyright (C) 2008 by Tobias Hunger <tobias@aquazul.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "dbustestcase.h"

#include <QtCore/QDebug>
#include <QtCore/QMetaObject>
#include <QtCore/QMetaMethod>

#include <QtCore/QCoreApplication>
#include <QtCore/QProcess>
#include <QtCore/QTemporaryFile>

#include <QtDBus/QDBusAbstractInterface>

#ifdef Q_OS_WIN
// nothing to do here...
#else
// Get unsetenv and kill functions and
// the kill function from the C headers:
extern "C"
{
#    include <stdlib.h>
#    include <sys/types.h>
#    include <signal.h>
}
#endif

// ---------------------------------------------------------------------------
// Helper code:
// ---------------------------------------------------------------------------

// Copied from Qt sources under the license of GPL:

struct QSignalSpyCallbackSet
{
    typedef void (*BeginCallback)(QObject *, int, void **);
    typedef void (*EndCallback)(QObject *, int);

    BeginCallback signal_begin_callback;
    BeginCallback slot_begin_callback;

    EndCallback signal_end_callback;
    EndCallback slot_end_callback;
};

void Q_CORE_EXPORT qt_register_signal_spy_callbacks(const QSignalSpyCallbackSet &);

// Private class:

class DBusTestCasePrivate
{
public:
    DBusTestCasePrivate(const bool priv_bus) :
        private_dbus(priv_bus)
    {
        QSignalSpyCallbackSet callbacks;
        callbacks.signal_begin_callback = &DBusTestCasePrivate::beginCallBackSignal;
        callbacks.slot_begin_callback = 0;
        callbacks.signal_end_callback = 0;
        callbacks.slot_end_callback = 0;

        qt_register_signal_spy_callbacks(callbacks);
    }

    ~DBusTestCasePrivate()
    { }

    static void beginCallBackSignal(QObject * caller,
                             int method_index,
                             void ** argv)
    {
        if (0 == qobject_cast<QDBusAbstractInterface*>(caller)) { return; }

        const QMetaObject * mo(caller->metaObject());
        QMetaMethod method(mo->method(method_index));

        DBusTestCase::DBusSignal signal;
        signal.time_stamp = QTime::currentTime();
        signal.object = caller;
        signal.signal = method.signature();

        QList<QByteArray> type_names;
        for (int i = 0; i < type_names.size(); ++i)
        {
            int type = QMetaType::type(type_names[i]);
            if (QMetaType::Void == type)
            { qFatal("Don't know how to handle type."); }
            signal.parameters << QVariant(type, argv[i + 1]);
        }
        signal_queue.append(signal);
    }

    void startDBus()
    {
#ifdef Q_OS_WIN
        // Windows does not allow to have more than one D-Bus daemon.
#else
        if (!private_dbus) { return; }

        // Start D-Bus process:
        QProcess dbus_proc;
        dbus_proc.start("dbus-launch");
        if (!dbus_proc.waitForStarted() || !dbus_proc.waitForFinished())
        { qFatal("Failed to start up private D-Bus session bus."); }

        if (dbus_proc.exitCode() != 0)
        { qFatal("Failed to set up private D-Bus session bus."); }

        // Read output of dbus-launch and store it into our environment:
        QByteArray line(dbus_proc.readLine());
        while(!line.isEmpty())
        {
            // Remove newline...
            line.chop(1);

            int index(line.indexOf('='));
            if (-1 == index)
            { qFatal("Failed to parse D-Bus session data."); }

            env_to_clean << line.mid(0, index);
            qDebug() << "    " << line.mid(0, index) << "="
                     << line.mid(index + 1);
            qputenv(line.mid(0, index).constData(), line.mid(index + 1));

            // read next line...
            line = dbus_proc.readLine();
        }

        // make sure we have a PID to kill:-)
        QVERIFY2(env_to_clean.contains(QByteArray("DBUS_SESSION_BUS_PID")),
                 "D-Bus did not set session bus PID.");
#endif

        // Start dbusviewer application (if set up in environment):
        QString dbusviewer_name(qgetenv("DBUSTEST_DBUSVIEWER"));
        if (!dbusviewer_name.isEmpty())
        {
            qDebug() << "Starting dbusviewer:" << dbusviewer_name;
            dbusviewer.start(dbusviewer_name);
        }
    }

    void stopDBus()
    {
#ifdef Q_OS_WIN
        // Windows does not allow to have more than one D-Bus daemon.
#else
        if (!private_dbus) { return; }

        bool ok(false);
        QByteArray value = qgetenv("DBUS_SESSION_BUS_PID");
        pid_t pid_to_kill(value.toInt(&ok));
        if (!ok) { qFatal("Failed to get PID of DBus session to kill."); }
        kill(pid_to_kill, 9);

        // clean up our environment again:
        foreach (QByteArray key, env_to_clean)
        { unsetenv(key.constData()); }
#endif

        // Stop dbusviewer (if it is running):
        if (dbusviewer.state() == QProcess::Running)
        {
            dbusviewer.terminate();
            if (!dbusviewer.waitForFinished())
            { dbusviewer.kill(); }
        }
    }

    const bool private_dbus;
    QList<QByteArray> env_to_clean;

    QProcess dbusviewer;

    QList<QProcess *> auts;

    static QList<DBusTestCase::DBusSignal> signal_queue;
};

QList<DBusTestCase::DBusSignal> DBusTestCasePrivate::signal_queue;

// ---------------------------------------------------------------------------
// DBusTestCase implementation:
// ---------------------------------------------------------------------------

DBusTestCase::DBusTestCase(bool private_dbus,
                           QObject * parent) :
    QObject(parent),
    d(new DBusTestCasePrivate(private_dbus))
{ Q_ASSERT(0 != d); }


DBusTestCase::~DBusTestCase()
{ delete d; }

QList<DBusTestCase::DBusSignal> DBusTestCase::recentSignals()
{
    QList<DBusTestCase::DBusSignal> result(d->signal_queue);
    d->signal_queue.clear();
    return result;
}

void DBusTestCase::startAUT(const QString & app, const QStringList & arguments)
{
    if (app.isEmpty()) { return; }

    QString app_path = QCoreApplication::applicationDirPath();
    QProcess * new_aut = new QProcess();

    new_aut->setWorkingDirectory(app_path);

    new_aut->start(app, arguments);
    if (!new_aut->waitForStarted()) { qFatal("Failed to start the AUT."); }

    // prepend so that we terminate the processes in the right order later:
    d->auts.prepend(new_aut);

    QVERIFY2(QProcess::Running == new_aut->state(),
             "AUT did not start up properly.");

    qDebug() << "AUT" << app << "started!";
}

void DBusTestCase::stopAUTs()
{
    foreach (QProcess * current_aut, d->auts)
    {
        QVERIFY2(QProcess::Running == current_aut->state(),
                 "AUT was no longer running when trying to shut it down.");

        current_aut->terminate();
        if (!current_aut->waitForFinished()) { current_aut->kill(); }

        delete current_aut;
    }
    d->auts.clear();
}


void DBusTestCase::initTestCase()
{ d->startDBus(); }

void DBusTestCase::cleanupTestCase()
{
    stopAUTs();
    d->stopDBus();
}
