# ui_cli_tests.py - unit tests for ui_cli.py
# Copyright (C) 2008, 2009  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import unittest

import computerjanitor
import computerjanitorapp


class MockCruft(object):

    def __init__(self, name):
        self._name = name
        self.cleaned = False
        self.post_cleaned = False
        
    def get_name(self):
        return self._name

    def cleanup(self):
        self.cleaned = True


class MockPlugin(object):

    def __init__(self, crufts):
        self._crufts = crufts
        self.post_cleaned = False
        
    def get_cruft(self):
        for cruft in self._crufts:
            yield cruft
        
    def post_cleanup(self):
        self.post_cleaned = True
            

class MockPluginManager(object):

    def __init__(self, plugins):
        self._plugins = plugins
        
    def get_plugins(self):
        return self._plugins


class MockState(object):

    def __init__(self, enabled):
        self._enabled = set(enabled)

    def is_enabled(self, name):
        return name in self._enabled

    def enable(self, name):
        if name not in self._enabled:
            self._enabled.add(name)

    def disable(self, name):
        if name in self._enabled:
            self._enabled.remove(name)

    def load(self, filename):
        self.load_filename = filename

    def save(self, filename):
        self.save_filename = filename


class MockOptionParser(object):

    def __init__(self):
        self.help_printed = False
        
    def print_help(self):
        self.help_printed = True


class MockApplication(object):

    def __init__(self, enabled):
        self.state = MockState(enabled)
        self.parser = MockOptionParser()

    def verify_apt_cache(self):
        pass

    def remove_whitelisted(self, crufts):
        return crufts


class MockOptions(object):

    def __init__(self):
        self.all = None
        self.state_file = "foo"
        self.no_act = None
        self.verbose = None


class CommandLineUserInterfaceTests(unittest.TestCase):

    def setUp(self):
        self.app = MockApplication(["foo"])
        self.cruft_names = ["foo", "bar"]
        self.crufts = [MockCruft(name) for name in self.cruft_names]
        self.cruftdict = dict((c.get_name(), c) for c in self.crufts)
        self.plugin = MockPlugin(self.crufts)
        self.pm = MockPluginManager([self.plugin])
        self.ui = computerjanitorapp.CommandLineUserInterface(self.app, 
                                          self.pm, mustberoot=False)

        self.options = MockOptions()

    def testInsistsOnBeingRoot(self):
        self.ui.mustberoot = True
        self.assertRaises(computerjanitor.Exception, self.ui.run, None, 
                          None)

    def testFindsTheRightCruft(self):
        self.assertEqual(self.ui.find_cruft(), self.crufts)

    def testShowsTheRightCruftTheRightWay(self):

        def mock_find_cruft():
            return self.crufts

        def mock_show_one_cruft(name, desc, state, width):
            output.append((name, state))

        output = []
        self.ui.find_cruft = mock_find_cruft
        self.ui.show_one_cruft = mock_show_one_cruft
        self.ui.show_cruft(MockOptions(), None)
        self.assertEqual(output, 
                         sorted([("foo", "removable"), ("bar", "ignored")]))

    def testIgnoresCruftCorrectly(self):
        self.ui.ignore(self.options, ["foo"])
        self.assertFalse(self.app.state.is_enabled("foo"))

    def testUnignoresCruftCorrectly(self):
        self.ui.unignore(self.options, ["bar"])
        self.assert_(self.app.state.is_enabled("bar"))

    def testCleansUpEnabledCruftWithDashDashAll(self):
        self.options.all = True
        self.ui.cleanup(self.options, [])
        self.assert_(self.cruftdict["foo"].cleaned)

    def testDoesNotCleanUpEnabledCruftWithDashDashAllWhenNoActIsSet(self):
        self.options.all = True
        self.options.no_act = True
        self.ui.cleanup(self.options, [])
        self.assertFalse(self.cruftdict["foo"].cleaned)

    def testCleansUpRequestedEnabledCruft(self):
        self.ui.cleanup(self.options, ["foo"])
        self.assert_(self.cruftdict["foo"].cleaned)

    def testDoesNotCleanUpDisaabledCruftWithDashDashAll(self):
        self.options.all = True
        self.ui.cleanup(self.options, [])
        self.assertFalse(self.cruftdict["bar"].cleaned)

    def testCleansUpRequestedDisabledCruft(self):
        self.ui.cleanup(self.options, ["bar"])
        self.assert_(self.cruftdict["bar"].cleaned)

    def testRunsPostCleanup(self):
        self.ui.cleanup(self.options, [])
        self.assert_(self.plugin.post_cleaned)

    def testDoesNotRunPostCleanupWhenNoActIsSet(self):
        self.options.no_act = True
        self.ui.cleanup(self.options, [])
        self.assertFalse(self.plugin.post_cleaned)

    def testRaisesExceptionForUnknownCruft(self):
        self.assertRaises(computerjanitor.Exception, self.ui.cleanup,
                          self.options, ["unknown"])

    def testHelpCallsParserPrintHelp(self):
        self.ui.help(None, None)
        self.assert_(self.app.parser.help_printed)

    def setup_run(self):
        names = ["show_cruft", "cleanup", "ignore", "unignore", "help"]
        for name in names:
            method = getattr(self.ui, name)
            wrapper = lambda options, args, name=name: \
                        setattr(self, "operation", name)
            setattr(self.ui, name, wrapper)

    def testRunCallsShowCruft(self):
        self.setup_run()
        self.ui.run(self.options, ["find"])
        self.assertEqual(self.operation, "show_cruft")

    def testRunCallsCleanup(self):
        self.setup_run()
        self.ui.run(self.options, ["cleanup"])
        self.assertEqual(self.operation, "cleanup")

    def testRunCallsIgnore(self):
        self.setup_run()
        self.ui.run(self.options, ["ignore"])
        self.assertEqual(self.operation, "ignore")

    def testRunCallsUnignore(self):
        self.setup_run()
        self.ui.run(self.options, ["unignore"])
        self.assertEqual(self.operation, "unignore")

    def testRunCallsHelp(self):
        self.setup_run()
        self.ui.run(self.options, ["help"])
        self.assertEqual(self.operation, "help")

    def testRunCallsHelpWhenThereAreNoArguments(self):
        self.setup_run()
        self.ui.run(self.options, [])
        self.assertEqual(self.operation, "help")

    def testRaisesExceptionForUnknownCommand(self):
        self.assertRaises(computerjanitor.Exception, self.ui.run, 
                          self.options, ["yikes"])

    def testRunLoadsStateFromRightFile(self):
        self.setup_run()
        self.ui.run(self.options, ["ignore"])
        self.assertEqual(self.app.state.load_filename, 
                         self.options.state_file)
