/*
    cobbler-enlist - Cobbler enlistment tool
    Copyright (C) 2011 Canonical Ltd.

    Authors: Dave Walker (Daviey) <DaveWalker@ubuntu.com>
             Carlos de-Avillez <carlos.de.avillez@ubuntu.com>
             Adam Gandelman <adamg@canonical.com>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, version 3 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

static int fault_occurred (xmlrpc_env * const envP) {
  if (!envP->fault_occurred)
    return 0;
  fprintf (stderr, "ERROR: %s (%d)\n", envP->fault_string, envP->fault_code);
  return 1;
}

int init_xmlrpc_env(struct cobbler_client *enlister) {
  if (debug) {
    printf("[DEBUG] Initializing xmlrpc enviornment.\n");
  }
  xmlrpc_env_init(&enlister->env);
  if(fault_occurred(&enlister->env) != 0)
    return 1;
  xmlrpc_client_setup_global_const(&enlister->env);
  return fault_occurred(&enlister->env);
}

int init_xmlrpc_client(struct cobbler_client *enlister) {
  if (debug)
    printf("[DEBUG] Initializing xmlrpc client.\n");
/*  xmlrpc_client_init2(&enlister->env,
                      XMLRPC_CLIENT_NO_FLAGS,
                      NAME, VERSION, NULL, 0);
  if (!fault_occured(&enlister->env))
    return 1;*/
  xmlrpc_client_create(&enlister->env,
                       XMLRPC_CLIENT_NO_FLAGS,
                       NAME, VERSION, 0, 0,
                       &enlister->client);
  return fault_occurred(&enlister->env);
}

int make_xmlrpc_call(struct cobbler_client *enlister,
                     char *method,
                     xmlrpc_value *arguments) {
  if (debug)
    printf("[DEBUG] Making xmlrpc call for method: %s\n", method);
  xmlrpc_server_info *serverinfo;
  serverinfo = xmlrpc_server_info_new(&enlister->env, enlister->serverUrl);
  xmlrpc_client_call2(&enlister->env,
                      enlister->client,
                      serverinfo,
                      method,
                      arguments,
                      &enlister->resultP);
  /* TODO: This causes error, find out why.
  xmlrpc_server_info_free(serverinfo); */
  return fault_occurred(&enlister->env);
}

int cobbler_login(struct cobbler_client *enlister) {
  if (debug)
    printf("[DEBUG] cobbler_login()\n");
  int rc = 0;
  xmlrpc_value *argument_array;
  xmlrpc_value *username;
  xmlrpc_value *password;
  argument_array = xmlrpc_array_new(&enlister->env);
  username = xmlrpc_string_new(&enlister->env, enlister->username);
  password = xmlrpc_string_new(&enlister->env, enlister->password);
  xmlrpc_array_append_item(&enlister->env, argument_array, username);
  xmlrpc_array_append_item(&enlister->env, argument_array, password);
  xmlrpc_DECREF(username);
  xmlrpc_DECREF(password);
  rc = make_xmlrpc_call(enlister, "login", argument_array);
  if (rc != 0)
    goto out;
  xmlrpc_read_string(&enlister->env, enlister->resultP, &enlister->token);
  if ((rc = fault_occurred(&enlister->env)) != 0)
    goto out;
  if (debug)
    printf("[DEBUG] The token is %s\n", enlister->token);
out:
  xmlrpc_DECREF(argument_array);
  if (debug)
    printf("[DEBUG] Login complete. rc=%d\n", rc);
  return rc;
}

int cobbler_register_system(struct cobbler_client *enlister) {
  if (debug)
    printf("[DEBUG] cobbler_register_system()\n");
  int rc = 0;

  /* at this point, everything we need should be containedin the enlister */
  xmlrpc_value *token = xmlrpc_string_new(&enlister->env, enlister->token);
  xmlrpc_value *name = xmlrpc_string_new(&enlister->env, enlister->name);
  xmlrpc_value *profile = xmlrpc_string_new(&enlister->env, enlister->profile);
  xmlrpc_value *macaddr = xmlrpc_string_new(&enlister->env, enlister->macaddr);

  /* used for forming modify_interface request */
  xmlrpc_value *macaddr_struct = xmlrpc_struct_new(&enlister->env);
  char macaddr_form_item[100];

  xmlrpc_value *argument_array;


  /* register a new server-side system object */
  argument_array = xmlrpc_array_new(&enlister->env);
  xmlrpc_array_append_item(&enlister->env, argument_array, token);
  if ((rc = make_xmlrpc_call(enlister, "new_system", argument_array)) != 0)
    goto out;
  xmlrpc_read_string(&enlister->env, enlister->resultP, &enlister->system);
  xmlrpc_value *system = xmlrpc_string_new(&enlister->env, enlister->system);
  if (debug)
    printf("[DEBUG] Registered new system: rc: %d system: %s\n", rc, enlister->system);

  /* modify the new system, set its name */
  argument_array = xmlrpc_array_new(&enlister->env);
  xmlrpc_array_append_item(&enlister->env, argument_array, system);
  xmlrpc_array_append_item(&enlister->env, argument_array,
                           xmlrpc_string_new(&enlister->env, "name"));
  xmlrpc_array_append_item(&enlister->env, argument_array, name);
  xmlrpc_array_append_item(&enlister->env, argument_array, token);
  if ((rc = make_xmlrpc_call(enlister, "modify_system", argument_array)) != 0)
    goto out;
  if (debug)
    printf("[DEBUG] Modified system name: %d\n", rc);

  /* also set its hostname to match its system name */
  argument_array = xmlrpc_array_new(&enlister->env);
  xmlrpc_array_append_item(&enlister->env, argument_array, system);
  xmlrpc_array_append_item(&enlister->env, argument_array,
                           xmlrpc_string_new(&enlister->env, "hostname"));
  xmlrpc_array_append_item(&enlister->env, argument_array, name);
  xmlrpc_array_append_item(&enlister->env, argument_array, token);
  if ((rc = make_xmlrpc_call(enlister, "modify_system", argument_array)) != 0)
    goto out;
  if (debug)
    printf("[DEBUG] Modified system hostname: %d\n", rc);

  /* assign a profile */
  argument_array = xmlrpc_array_new(&enlister->env);
  xmlrpc_array_append_item(&enlister->env, argument_array, system);
  xmlrpc_array_append_item(&enlister->env, argument_array,
                           xmlrpc_string_new(&enlister->env, "profile"));
  xmlrpc_array_append_item(&enlister->env, argument_array, profile);
  xmlrpc_array_append_item(&enlister->env, argument_array, token);
  if ((rc = make_xmlrpc_call(enlister, "modify_system", argument_array)) != 0)
    goto out;
  if (debug)
  printf("[DEBUG] Modified system profile: %d\n", rc);

  /* set its mac address */
  sprintf(macaddr_form_item, "macaddress-%s", enlister->netif);
  xmlrpc_struct_set_value(&enlister->env, macaddr_struct,
                          macaddr_form_item, macaddr);
  argument_array = xmlrpc_array_new(&enlister->env);
  xmlrpc_array_append_item(&enlister->env, argument_array, system);
  xmlrpc_array_append_item(&enlister->env, argument_array,
                           xmlrpc_string_new(&enlister->env, "modify_interface"));
  xmlrpc_array_append_item(&enlister->env, argument_array, macaddr_struct);
  xmlrpc_array_append_item(&enlister->env, argument_array, token);
  if ((rc = make_xmlrpc_call(enlister, "modify_system", argument_array)) != 0)
    goto out;
  if (debug)
    printf("[DEBUG] Modified interface: %d\n", rc);

  /* assign a mgmt class */
  if (enlister->mgmt_class) {
    xmlrpc_value *mgmt_class = xmlrpc_string_new(&enlister->env,
                                                      enlister->mgmt_class);
    argument_array = xmlrpc_array_new(&enlister->env);
    xmlrpc_array_append_item(&enlister->env, argument_array, system);
    xmlrpc_array_append_item(&enlister->env, argument_array,
                             xmlrpc_string_new(&enlister->env, "mgmt_classes"));
    xmlrpc_array_append_item(&enlister->env, argument_array, mgmt_class);
    xmlrpc_array_append_item(&enlister->env, argument_array, token);
    if ((rc = make_xmlrpc_call(enlister, "modify_system", argument_array)) != 0)
      xmlrpc_DECREF(mgmt_class);
      goto out;
    if (debug)
      printf("[DEBUG] Modified mgmt class: %d\n", rc);
    xmlrpc_DECREF(mgmt_class);
  }

out:
  xmlrpc_DECREF(token);
  xmlrpc_DECREF(name);
  xmlrpc_DECREF(profile);
  xmlrpc_DECREF(system);
  xmlrpc_DECREF(macaddr);
  xmlrpc_DECREF(macaddr_struct);
  xmlrpc_DECREF(argument_array);
  return rc;
}

int cobbler_save_system(struct cobbler_client *enlister) {
  if (debug)
    printf("[DEBUG] cobbler_save_system()\n");
  int rc = 0;
  xmlrpc_value *argument_array;
  argument_array = xmlrpc_array_new(&enlister->env);
  xmlrpc_value *token = xmlrpc_string_new(&enlister->env, enlister->token);
  xmlrpc_value *system = xmlrpc_string_new(&enlister->env, enlister->system);
  xmlrpc_array_append_item(&enlister->env, argument_array, system);
  xmlrpc_array_append_item(&enlister->env, argument_array, token);
  if ((rc = make_xmlrpc_call(enlister, "save_system", argument_array)) != 0)
    goto out;
  printf("[DEBUG] System saved.\n");
out:
  xmlrpc_DECREF(token);
  xmlrpc_DECREF(system);
  xmlrpc_DECREF(argument_array);
  return rc;
}
