/*
    cobbler-enlist - Cobbler enlistment tool
    Copyright (C) 2011 Canonical Ltd.

    Authors: Dave Walker (Daviey) <DaveWalker@ubuntu.com>
             Carlos de-Avillez <carlos.de.avillez@ubuntu.com>
             Adam Gandelman <adamg@canonical.com>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, version 3 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

    Usage: ./cobbler-enlist --serverurl=http://cobbler.server/cobbler_api \
             --username=cobbler --password=cobbler --name=test \
             --profile=ubuntu-server-x86_64
            optionally, '--netif=<network I/F>' can also be passed, restricting
            registration to this I/F. If no --netif, then all I/Fs will be
            registered.

*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdbool.h>

#include <xmlrpc-c/base.h>
#include <xmlrpc-c/client.h>

#include <net/if.h>
#include <sys/ioctl.h>
#include <netinet/if_ether.h>

#include <getopt.h>

#include "cobbler-enlist.h"
#include "cobbler-xmlrpc.h"

int
main (int const argc, char **argv)
{
  int rc;

  struct cobbler_client *enlister;
  enlister = (struct cobbler_client *) malloc(sizeof(struct cobbler_client));
  init_client(enlister);
  int c;

  //int this_option_optind = optind ? optind : 1;
  int option_index = 0;

  const char* const short_options = "hds:u:p:n:P:i:m:";
  static struct option long_options[] = {
    {"serverurl", 1, 0, 's'},
    {"username", 1, 0, 'u'},
    {"password", 1, 0, 'p'},
    {"name", 1, 0, 'n'},
    {"profile", 1, 0, 'P'},
    {"netif", 1, 0, 'i'},
    {"manclass", 1, 0, 'm'},
    {"help", 0, 0, 'h'},
    //{"default-interface", 1, 0, 0},
    {0, 0, 0, 0}
  };

  if (getenv("COBBLER_ENLIST_DEBUG"))
    debug = 1;

  while (1)
    {
      c = getopt_long (argc, argv, short_options, long_options, &option_index);
      if (c == -1)
        break;
      if (c == '?') {
        help(argv[0]);
        exit(1);
      }

      switch (c)
        {
        case '?':
          help(argv[0]);
          exit(1);

        case 's':
          enlister->serverUrl = optarg;
          break;

        case 'u':
          enlister->username = optarg;
          break;

        case 'p':
          enlister->password = optarg;
          break;

        case 'n':
          enlister->name = optarg;
          break;

        case 'P':
          enlister->profile = optarg;
          break;

        case 'i':
          enlister->netif = optarg;
          break;

        case 'm':
          enlister->mgmt_class = optarg;
          break;

        case 'h':
          help(argv[0]);
          exit(0);
          break;

        default:
          break;
        }
    }

  if (optind < argc)
    {
      printf ("[%s]non-option ARGV-elements: ", argv[0]);
      while (optind < argc)
      {
        printf ("%s ", argv[optind++]);
      }
      printf ("\n");
    }

  /* check if all required parameters have been passed  */
  if ((enlister->serverUrl == NULL) ||
      (enlister->username == NULL) ||
      (enlister->password == NULL) ||
      (enlister->name == NULL) ||
      (enlister->profile == NULL))
  {
    fprintf(stderr, "[%s] ERROR: not all required parameters were passed\n", argv[0]);
    help(argv[0]);
    exit (1);
  }


  if ((rc = get_mac_address(enlister)) != 0) {
    printf("ERROR: Could not determine mac address for any interface on system.\n");
    exit(1);
  }

  if (debug)
    display_config(enlister);

  /* Initialize our error-handling environment. */
  if ((rc = init_xmlrpc_env(enlister)) != 0)
    goto out;
  if ((rc = init_xmlrpc_client(enlister)) != 0)
    goto out;
  if ((rc = cobbler_login(enlister)) != 0)
    goto out;
  if ((rc =  cobbler_register_system(enlister)) != 0)
    goto out;
  if ((rc = cobbler_save_system(enlister)) != 0)
    goto out;

  if (debug)
    printf("[DEBUG] Enlistment complete\n");

out:
  if (debug)
    printf("[DEBUG] Exit: %d\n", rc);
  xmlrpc_env_clean(&enlister->env);
  return rc;
}

