/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2007  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>

#include <gtk/gtk.h>

#include "export.h"
#include "tracer.h"

static gboolean write_btsnoop(GtkTreeModel *model, GtkTreePath *path,
					GtkTreeIter *iter, gpointer user_data)
{
	GIOChannel *io = user_data;
	gsize bytes_written;
	struct btsnoop_pkt pkt;
	guint layer, type, size;
	gpointer data;
	guint64 timestamp, ts;

	gtk_tree_model_get(model, iter,
				COLUMN_LAYER, &layer, COLUMN_TYPE, &type,
				COLUMN_SIZE, &size, COLUMN_DATA, &data,
				COLUMN_TIMESTAMP, &timestamp, -1);

	if (layer != LAYER_HCI)
		return FALSE;

	memset(&pkt, 0, sizeof(pkt));

	switch (type) {
	case 0x00:
		pkt.flags = GUINT32_TO_BE(0x02);
		break;
	case 0x01:
		pkt.flags = GUINT32_TO_BE(0x03);
		break;
	case 0x02:
		pkt.flags = GUINT32_TO_BE(0x00);
		break;
	case 0x03:
		pkt.flags = GUINT32_TO_BE(0x01);
		break;
	default:
		return FALSE;
	}

	ts = timestamp;
	ts = ((ts >> 32) - 946684800ll) * 1000000ll + (ts & 0xffffffff);

	pkt.size  = GUINT32_TO_BE((guint32) size);
	pkt.len   = pkt.size;
	pkt.drops = 0;
	pkt.ts    = GUINT64_TO_BE(ts + 0x00E03AB44A676000ll);

	g_io_channel_write_chars(io, (void *) &pkt, BTSNOOP_PKT_SIZE,
							&bytes_written, NULL);
	if (bytes_written != BTSNOOP_PKT_SIZE)
		return TRUE;

	g_io_channel_write_chars(io, data, size, &bytes_written, NULL);
	if (bytes_written != size)
		return TRUE;

	return FALSE;
}

static gboolean write_pktlog(GtkTreeModel *model, GtkTreePath *path,
					GtkTreeIter *iter, gpointer user_data)
{
	GIOChannel *io = user_data;
	gsize bytes_written;
	struct pktlog_hdr pkt;
	guint layer, type, size;
	gpointer data;
	guint64 timestamp;

	gtk_tree_model_get(model, iter,
				COLUMN_LAYER, &layer, COLUMN_TYPE, &type,
				COLUMN_SIZE, &size, COLUMN_DATA, &data,
				COLUMN_TIMESTAMP, &timestamp, -1);

	if (layer != LAYER_HCI)
		return FALSE;

	memset(&pkt, 0, sizeof(pkt));
	pkt.len  = GUINT32_TO_BE(size + 9);
	pkt.ts   = GUINT64_TO_BE(timestamp);
	pkt.type = type;

	g_io_channel_write_chars(io, (void *) &pkt, PKTLOG_HDR_SIZE,
							&bytes_written, NULL);
	if (bytes_written != PKTLOG_HDR_SIZE)
		return TRUE;

	g_io_channel_write_chars(io, data, size, &bytes_written, NULL);
	if (bytes_written != size)
		return TRUE;

	return FALSE;
}

int export_model(GtkTreeModel *model, const char *filename)
{
	GIOChannel *io;
	guint8 format = FORMAT_UNKNOWN;

	if (g_str_has_suffix(filename, ".log") == TRUE)
		format = FORMAT_BTSNOOP;

	if (g_str_has_suffix(filename, ".btsnoop") == TRUE)
		format = FORMAT_BTSNOOP;

	if (g_str_has_suffix(filename, ".pklg") == TRUE)
		format = FORMAT_PKTLOG;

	if (format == FORMAT_UNKNOWN)
		return -1;

	io = g_io_channel_new_file(filename, "w", NULL);
	if (io == NULL)
		return -1;

	g_io_channel_set_close_on_unref(io, TRUE);
	g_io_channel_set_encoding(io, NULL, NULL);
	g_io_channel_set_buffered(io, FALSE);

	if (format == FORMAT_BTSNOOP) {
		struct btsnoop_hdr hdr;
		gsize bytes_written;

		memset(&hdr, 0, sizeof(hdr));
		memcpy(hdr.id, btsnoop_id, sizeof(btsnoop_id));
		hdr.version = GUINT32_TO_BE(1);
		hdr.type = GUINT32_TO_BE(1001);

		g_io_channel_write_chars(io, (void *) &hdr, BTSNOOP_HDR_SIZE,
							&bytes_written, NULL);
		if (bytes_written != BTSNOOP_HDR_SIZE) {
			g_io_channel_unref(io);
			return -1;
		}

		gtk_tree_model_foreach(model, write_btsnoop, io);
	}
		gtk_tree_model_foreach(model, write_pktlog, io);

	g_io_channel_unref(io);

	return 0;
}
