/* build-revision.c:
 *
 * vim:smartindent ts=8:sts=2:sta:et:ai:shiftwidth=2
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "hackerlab/bugs/panic.h"
#include "hackerlab/mem/alloc-limits.h"
#include "hackerlab/mem/mem.h"
#include "hackerlab/char/str.h"
#include "hackerlab/fs/file-names.h"
#include "hackerlab/vu/safe.h"
#include "libfsutils/copy-file.h"
#include "libfsutils/dir-listing.h"
#include "libfsutils/tmp-files.h"
#include "libfsutils/rmrf.h"
#include "libfsutils/link-tree.h"
#include "libfsutils/ensure-dir.h"
#include "libarch/ancestry.h"
#include "libarch/arch.h"
#include "libarch/namespace.h"
#include "libarch/libraries.h"
#include "libarch/project-tree.h"
#include "libarch/archive.h"
#include "libarch/archives.h"
#include "libarch/apply-changeset.h"
#include "libarch/pristines.h"
#include "libarch/invent.h"
#include "libarch/inode-sig.h"
#include "libarch/chatter.h"
#include "libarch/patch-logs.h"
#include "libarch/library-txn.h"
#include "libarch/build-revision.h"



static int
build_revision (int chatter_fd,
                int * patch_list_length,
		t_uchar * dest_dir,
                struct arch_archive * arch,
                t_uchar * archive, 
		t_uchar * revision,
                t_uchar * cache_dir,
		int try_link);

static void
select_patch_list (rel_table * patch_list, rel_table candidate);

static void 
make_patch_list (int chatter_fd,
                 rel_table * patch_list,
		 t_uchar * ancestor_archive, 
		 t_uchar * ancestor_revision, 
                 t_uchar * descendant_archive, 
		 t_uchar * descendant_revision,
		 assoc_table  contin_list,
		 struct arch_archive *arch);

static int 
patch_direction (rel_table patch_list, t_uchar *archive, t_uchar *revision);

static int 
decrement_revision (int chatter_fd,
                    struct arch_archive ** my_arch, 
                    t_uchar ** my_archive, 
		    t_uchar ** my_revision, 
		    assoc_table contin_list,
		    struct arch_archive * arch, 
		    t_uchar * archive, 
		    t_uchar * revision);

static void
find_patchlists (rel_table * loc_patches,
		 rel_table * arch_patches,
		 int chatter_fd,  
		 t_uchar * archive, 
		 t_uchar *revision,
		 struct arch_archive *arch,
		 assoc_table * contin_list,
		 t_uchar * cache_dir);

static int
direct_descendant (struct arch_archive * arch, t_uchar * revision, 
                   t_uchar * candidate_descendant);


static t_uchar * 
find_previous_library (int chatter_fd, t_uchar * archive, t_uchar * revision);

static int
find_multisource (t_uchar ** lib_rev, 
		  t_uchar ** arch_rev, 
		  t_uchar ** continuation, 
		  int chatter_fd,
		  struct arch_archive * arch, 
		  t_uchar * revision,
		  t_uchar * cache_dir);

static t_uchar *
scan_archive (enum arch_revision_type *type,
              int * num_scanned,
              t_uchar ** newest_cached, 
              t_uchar ** oldest_cached, 
	      struct arch_archive *arch, 
	      t_uchar * start_revision, 
	      t_uchar * end_revision);

static void 
find_prev_revision (assoc_table  * contin_list,
		    t_uchar ** local_rev, 
		    t_uchar ** arch_rev, 
		    int chatter_fd,
		    struct arch_archive * arch, 
		    t_uchar * start_revision,
		    t_uchar * cache_dir);

static struct arch_archive * archive_connect_soft (t_uchar * archive);

int find_library_forwards (int chatter_fd, t_uchar ** found_archive, t_uchar ** found_revision, t_uchar * archive, t_uchar * revision);

static int 
copy_local_revision (int chatter_fd, 
		     t_uchar * dest_dir,  
		     t_uchar * archive, 
		     t_uchar * revision, 
		     t_uchar * cache_dir,
		     int try_link);

static int 
copy_library_revision (int chatter_fd, 
		       t_uchar * dest_dir, 
		       t_uchar * archive,
		       t_uchar * revision, 
		       int try_link);

static int 
copy_pristine_revision (int chatter_fd, 
			t_uchar * dest_dir,  
			t_uchar * archive, 
			t_uchar * revision, 
			t_uchar * cache_dir);

static void
move_tmp_revision (t_uchar * dest_dir, t_uchar * tmpdir);

static int 
copy_archive_revision(int chatter_fd,
		      t_uchar * dest_dir,
		      struct arch_archive *arch,
		      t_uchar * revision);

static void 
copy_cached_revision (int chatter_fd, 
		      t_uchar * dest_dir, 
		      struct arch_archive * arch, 
		      t_uchar * revision);

static void
copy_import_revision (int chatter_fd, 
		      t_uchar * dest_dir, 
		      struct arch_archive * arch, 
		      t_uchar * revision);

static t_uchar * 
full_revision (t_uchar * archive, t_uchar * revision);

static int
find_for_previous_revision (int chatter_fd,
                            struct arch_archive ** prev_arch, 
                            t_uchar ** prev_archive, 
			    t_uchar ** prev_revision, 
			    assoc_table contin_list,
			    struct arch_archive * arch, 
			    t_uchar * archive,
			    t_uchar * revision);

static void
apply_revision_patch (int chatter_fd, 
		      t_uchar * dest_dir, 
		      struct arch_archive * arch, 
		      t_uchar * revision,
		      int reverse);

static void 
apply_revisions (int chatter_fd, 
                 t_uchar * dest_dir, 
		 struct arch_archive * arch, 
		 rel_table patch_list,
		 int apply_direction);

/* return non-0 if it came from a library.
 */
int
arch_build_revision (int chatter_fd,
                     t_uchar * dest_dir,
                     struct arch_archive * arch,
                     t_uchar * archive, t_uchar * revision,
                     t_uchar * cache_dir)
{
  struct arch_archive * my_arch = arch;
  int from_library = 0;
  int n_patches;
  if (my_arch == 0)
    my_arch = arch_archive_connect(archive, 0);
  invariant (my_arch != 0);

  from_library = build_revision (chatter_fd, &n_patches, dest_dir, my_arch, archive, revision, cache_dir, 0) == 1;
  if (n_patches > 1) from_library = 0;
  if (my_arch != arch) 
    arch_archive_close (my_arch);
  return from_library;
}

void
arch_build_library_revision(int chatter_fd,
                            int no_patch_noise, 
                            t_uchar * dest_directory,
                            struct arch_archive * arch, 
                            t_uchar * revision,
                            t_uchar * opt_library, 
                            t_uchar * opt_same_dev_path,
                            int sparse,
                            arch_apply_changeset_report_callback callback,
			    int escape_classes)
{
  t_uchar * fq_ancestor_rev = 0;
  t_uchar * previous_revision = 0;
  t_uchar * ancestor_archive = 0;
  t_uchar * ancestor_revision = 0;
  t_uchar * library_ancestor = 0;
  t_uchar * dest_dir_dir = 0;
  t_uchar * tmp_dir = 0;
  t_uchar * patch_set_dir = 0;
  t_uchar * log_path = 0;
  t_uchar * copy_log_path = 0;
  t_uchar * ancestor_rec_path = 0;
  t_uchar * prev_rec_path = 0;

    {
       t_uchar *fq_current = arch_fully_qualify (arch->official_name, revision);
       fq_ancestor_rev = arch_patch_ancestor (fq_current, 0, 0);
       lim_free (0, fq_current);
    }

  previous_revision = arch_previous_revision (arch, revision);

  dest_dir_dir = file_name_directory_file (0, dest_directory);
  ensure_directory_exists (dest_dir_dir);
  tmp_dir = tmp_file_name (dest_dir_dir, ",,new-revision");  
  
  if (fq_ancestor_rev)
    {
      ancestor_archive = arch_parse_package_name (arch_ret_archive, 0, fq_ancestor_rev);
      ancestor_revision = arch_parse_package_name (arch_ret_non_archive, 0, fq_ancestor_rev);
      library_ancestor = arch_library_find_picky (ancestor_archive, ancestor_revision, 1, dest_dir_dir, 0);
    }

  if (library_ancestor)
    {
      /* If the immediate ancestor exists in the same library,
       * then fill our tmp dir with a link-tree to that ancestor.
       * It will be patched for our target revision later.
       */

      arch_chatter (chatter_fd, "* found immediate ancestor revision in library (%s/%s)\n", ancestor_archive, ancestor_revision);
      build_link_tree (library_ancestor, tmp_dir);
    }
  else if (!fq_ancestor_rev || str_cmp (ancestor_archive, arch->official_name))
    {
      /* If there is no immediate ancestor (we're trying to build 
       * an import revision) or if the ancestor is in some other archive,
       * fill the tmp-dir with the target revision just by building
       * it from scratch.
       * 
       * There's no need to further patch the tmp-dir after that.
       *
       * WARNING: tag patches are not reversible in older 
       * arch archives, so do not consider removing the cross-archive limit.
       */
      int n_patches;

      safe_mkdir (tmp_dir, 0777);
      build_revision (chatter_fd, &n_patches, tmp_dir, arch, arch->official_name, revision, 0, 1);

      patch_set_dir = file_name_in_vicinity (0, tmp_dir, ",,patch-set");
      rmrf_file (patch_set_dir);

      if (fq_ancestor_rev)
        {
          arch_get_patch (arch, revision, patch_set_dir);
        }
      else
        {
          safe_mkdir (patch_set_dir, 0777);
        }

      goto no_need_to_patch;
    }
  else if (!sparse)
    {
      /* There is an ancestor, it's in the same archive, it's not 
       * in the library, and we're not building a --sparse library.
       * So, recursively add the ancestor to the library and then
       * build a link-tree into tmp-dir.   We'll patch tmp-dir to the
       * target revision later.
       */
      int n_patches;

      arch_chatter (chatter_fd, "* recursively adding %s/%s to library\n", ancestor_archive, ancestor_revision);
      arch_library_add (chatter_fd, no_patch_noise, arch, ancestor_revision, opt_library, opt_same_dev_path, 0, escape_classes);
      library_ancestor = arch_library_find_picky (ancestor_archive, ancestor_revision, 1, dest_dir_dir, 0);
      if (library_ancestor)
	build_link_tree (library_ancestor, tmp_dir);
      else 
	build_revision (chatter_fd, &n_patches, tmp_dir, arch, ancestor_archive, ancestor_revision, 0, 1);
    }
  else
    {
      /* There is an ancestor, it's in the same archive, it's
       * not in the library, but we're building a --sparse library.
       *
       * If some _earlier_ ancestor is at hand, link to that and then
       * patch.   Otherwise, build the target revision from scratch.
       * 
       * Either way, at the end, tmp-dir will have the target revision
       * and there's no need to further patch it.
       */
      /* list of patches to apply to revision in library to achieve new revision
       */
      rel_table revisions = 0;
      int n_patches;


      arch_chatter (chatter_fd, "* searching ancestor revision in library in archive %s\n", ancestor_archive);
     
      /*  build the revision from scratch
       */
      safe_mkdir (tmp_dir, 0777);
      build_revision (chatter_fd, &n_patches, tmp_dir, arch, arch->official_name, revision, 0, 1);

      /* keep last patchset inside revision
       */
      patch_set_dir = file_name_in_vicinity (0, tmp_dir, ",,patch-set");
      rmrf_file (patch_set_dir);
      arch_get_patch (arch, revision, patch_set_dir);

      rel_free_table (revisions);
      goto no_need_to_patch;
    }


  /* This is reached only if we have one-last patch to make to 
   * tmp-dir to achieve the target revision.
   * 
   * If tmp-dir already has the target, then we `goto no_need_to_patch',
   * below.
   */

  patch_set_dir = file_name_in_vicinity (0, tmp_dir, ",,patch-set");
  rmrf_file (patch_set_dir);
  arch_get_patch (arch, revision, patch_set_dir);

  {
    struct arch_apply_changeset_report report = {0,};

    arch_chatter (chatter_fd, "* patching for this revision (%s/%s)\n", arch->official_name, revision);

    if ((chatter_fd >= 0) && !no_patch_noise)
      {
        report.callback = callback;
        report.thunk = (void *)(long)chatter_fd;
      }

    arch_apply_changeset (&report, patch_set_dir, tmp_dir, arch_unspecified_id_tagging, arch_inventory_unrecognized, 0, 0, escape_classes);
    invariant (!arch_conflicts_occurred (&report));

    arch_free_apply_changeset_report_data (&report);
  }

 no_need_to_patch:

  /* Finish up adding ancillary files to the tmp-dir and then
   * install it in the library.
   */

  log_path = arch_log_file (tmp_dir, arch->official_name, revision);
  copy_log_path = file_name_in_vicinity (0, patch_set_dir, "=log.txt");
  copy_file (log_path, copy_log_path);

  ancestor_rec_path = file_name_in_vicinity (0, patch_set_dir, "=ancestor");
  prev_rec_path = file_name_in_vicinity  (0, patch_set_dir, "=previous");

  {
    int out_fd;

    out_fd = safe_open (ancestor_rec_path, O_WRONLY | O_CREAT | O_EXCL, 0444);
    safe_printfmt (out_fd, "%s\n", fq_ancestor_rev);
    safe_close (out_fd);

    out_fd = safe_open (prev_rec_path, O_WRONLY | O_CREAT | O_EXCL, 0444);
    safe_printfmt (out_fd, "%s\n", previous_revision);
    safe_close (out_fd);
  }

  {
    rel_table index = 0;
    t_uchar * index_by_name_path = 0;
    t_uchar * index_path = 0;
    int index_by_name_fd = -1;
    int index_fd = -1;

    {
      arch_patch_id patch_id;
    
      arch_patch_id_init_archive (&patch_id, arch->official_name, revision);
      arch_set_tree_version (tmp_dir, arch_patch_id_branch (&patch_id));

      arch_patch_id_finalise (&patch_id);
    }

    index = arch_source_inventory (tmp_dir, 1, 0, 0);
    index_by_name_path = file_name_in_vicinity (0, tmp_dir, ",,index-by-name");
    index_path = file_name_in_vicinity (0, tmp_dir, ",,index");

    rmrf_file (index_by_name_path);
    rmrf_file (index_path);

    index_by_name_fd = safe_open (index_by_name_path, O_WRONLY | O_CREAT | O_EXCL, 0444);
    rel_print_pika_escape_iso8859_1_table (index_by_name_fd, arch_escape_classes, index);
    safe_close (index_by_name_fd);

    rel_sort_table_by_field (0, index, 1);
    index_fd = safe_open (index_path, O_WRONLY | O_CREAT | O_EXCL, 0444);
    rel_print_pika_escape_iso8859_1_table (index_fd, arch_escape_classes, index);
    safe_close (index_fd);

    rel_free_table (index);
    lim_free (0, index_by_name_path);
    lim_free (0, index_path);
  }

  arch_snap_inode_sig (tmp_dir, arch->official_name, revision);
  /* tree-version set up above */

  safe_rename (tmp_dir, dest_directory);

  lim_free (0, fq_ancestor_rev);
  lim_free (0, previous_revision);
  lim_free (0, ancestor_archive);
  lim_free (0, ancestor_revision);
  lim_free (0, library_ancestor);
  lim_free (0, dest_dir_dir);
  lim_free (0, tmp_dir);
  lim_free (0, patch_set_dir);
  lim_free (0, ancestor_rec_path);
  lim_free (0, prev_rec_path);

}


/*
 * The selection code is built on the following false assumptions:
 * 1 all patches are the same size
 * 2 cached trees take around as long to download as 16 patches
 * 3 throughput is the same for all archives
 * 
 * This is because we don't know the truth.  But
 * 1 on *average*, this is true (by definition).
 * 2 value reached by experimentation w/ a particular tree
 * 3 this a common case.
 */
static int
build_revision (int chatter_fd,
                int * n_patches,
                     t_uchar * dest_dir,
                     struct arch_archive * arch,
                     t_uchar * archive, 
		     t_uchar * revision,
                     t_uchar * cache_dir,
		     int try_link)
{
  struct arch_archive * my_arch = arch;
  t_uchar * my_archive = archive;
  t_uchar * my_revision = revision;
  int revision_found = 0;
  int from_library = 0;
  int archive_revision_cost = 50;
  assoc_table contin_list = 0;

  rel_table loc_patches = 0;
  rel_table arch_patches = 0;
  rel_table patch_list = 0;
  int status = 0;

  find_patchlists (&loc_patches, &arch_patches, chatter_fd, archive, revision, arch, &contin_list, cache_dir);

  if (rel_n_records(loc_patches)>0 || rel_n_records(arch_patches) >0)
    revision_found = 1;
  else
    {
      panic ("The requested revision cannot be built.  It is based on archives that are not registered.");
    }
  
  if (revision_found)
    {
      int n_arch_patches;
      patch_list = loc_patches;
      *n_patches = rel_n_records (patch_list);
      n_arch_patches = rel_n_records (arch_patches);
      if (*n_patches == 0 || (n_arch_patches > 0 && (*n_patches > n_arch_patches + archive_revision_cost)))
	{
	  patch_list = arch_patches;
	  rel_free_table (loc_patches);
	  *n_patches = rel_n_records(patch_list);
	}
      else
	{
	  rel_free_table (arch_patches);
	}

      if (patch_direction(patch_list, archive, revision) == -1)
        {
          my_archive = (patch_list)[*n_patches -1][0];
          my_revision = (patch_list)[*n_patches -1][1];
        }
      else 
        {
          my_archive = (patch_list)[0][0];
          my_revision = (patch_list)[0][1];
        }
      if (patch_list == arch_patches)
	{
	  struct arch_archive * tmp_arch = arch;
	  if (str_cmp (arch->official_name, my_archive))
	    tmp_arch = arch_archive_connect (my_archive, 0);
	      
	  status = copy_archive_revision (chatter_fd, dest_dir, tmp_arch, my_revision);
	  if (tmp_arch != arch)
	    arch_archive_close (tmp_arch);
	}
      else
	{
	  status = copy_local_revision (chatter_fd, dest_dir, my_archive, my_revision, cache_dir, try_link);
	  invariant (status != 0);
	  from_library = (status == 2);
	}
    }
  else
    {
      make_patch_list(chatter_fd, &patch_list, my_archive, my_revision, archive, revision, contin_list, arch);
    }
  if (my_arch != arch)
    arch_archive_close (my_arch);

  apply_revisions (chatter_fd, dest_dir, arch, patch_list, patch_direction (patch_list, archive, revision));
  rel_free_table (patch_list);
  arch_snap_inode_sig (dest_dir, archive, revision);
    {
      arch_patch_id patch_id;
      
      arch_patch_id_init_archive (&patch_id, archive, revision);
      arch_set_tree_version (dest_dir, arch_patch_id_branch (&patch_id));

      arch_patch_id_finalise (&patch_id);
    }
  
  return status;
}

static void
select_patch_list (rel_table * patch_list, rel_table candidate)
{
  int candidate_n = rel_n_records (candidate);
  int current_n = rel_n_records (*patch_list);
  if (candidate_n < 1) 
    {
      rel_free_table (candidate);
      return;
    }
  else if (current_n < 1)
    {
      rel_free_table (*patch_list);
      * patch_list = candidate;
    }
  else if (current_n > candidate_n)
    {
      rel_free_table (*patch_list);
      * patch_list = candidate;
    }
}

static void 
make_patch_list (int chatter_fd,
                 rel_table * patch_list, 
		 t_uchar * ancestor_archive, 
		 t_uchar * ancestor_revision, 
                 t_uchar * descendant_archive, 
		 t_uchar * descendant_revision, 
		 assoc_table contin_list,
		 struct arch_archive * arch)
{
  t_uchar * my_archive = str_save (0, descendant_archive);
  t_uchar * my_revision = str_save (0, descendant_revision);
  struct arch_archive * my_arch = arch;
  int first_loop = 1;
  int entries = 0;

  

  while (str_cmp (my_archive, ancestor_archive) || str_cmp (my_revision, ancestor_revision))
    {
      if (first_loop)
	{
	  arch_chatter (chatter_fd, "* building revision list from %s/%s to %s/%s\n", ancestor_archive, ancestor_revision, descendant_archive, descendant_revision);
	  first_loop = 0;
	}
      rel_add_records (patch_list, rel_make_record (my_archive, my_revision, 0), 0);
      if (decrement_revision (chatter_fd, &my_arch, &my_archive, &my_revision, contin_list, arch, descendant_archive, descendant_revision) == -1)
      {
	 rel_free_table (*patch_list);
	 *patch_list = 0;
	 if (my_archive != descendant_archive)
	    lim_free (0, my_archive);
      }
      
    }
  rel_add_records (patch_list, rel_make_record (my_archive, my_revision, 0), 0);
  lim_free (0, my_archive);
  lim_free (0, my_revision);
  if (my_arch != arch)
     arch_archive_close (my_arch);
  entries = rel_n_records(*patch_list);
  if (entries > 1) 
    arch_chatter (chatter_fd, "* %d entries added to revision list\n", entries);
  rel_reverse_table (*patch_list);
}


static int 
patch_direction (rel_table patch_list, t_uchar *archive, t_uchar *revision)
{
  int apply_direction = 1;
  if (rel_n_records(patch_list) > 0)
  {
    if (!str_cmp(archive, patch_list[0][0]) && !str_cmp(patch_list[0][1], revision))
      apply_direction = -1;
  }
  return apply_direction;
}

static int 
decrement_revision (int chatter_fd,
                    struct arch_archive ** my_arch, 
                    t_uchar ** my_archive, 
                    t_uchar ** my_revision,
		    assoc_table contin_list,
                    struct arch_archive * arch, 
                    t_uchar * archive, 
                    t_uchar * revision)
{
  struct arch_archive *prev_arch = 0;
  t_uchar *prev_archive = 0;
  t_uchar *prev_revision = 0;

  if (find_for_previous_revision (chatter_fd, &prev_arch, &prev_archive, &prev_revision, contin_list, *my_arch, *my_archive, *my_revision) == -1)
    return -1;

  if (*my_arch != prev_arch)
    {
      if (arch != *my_arch)
	arch_archive_close (*my_arch);
      *my_arch = prev_arch;
    }
  if (*my_archive != prev_archive)
    {
      if (archive != *my_archive)
	lim_free (0, *my_archive);
      *my_archive = prev_archive;
    }
  if (*my_revision != prev_revision)
    {
      if (revision != *my_revision)
	lim_free (0, *my_revision);
      *my_revision = prev_revision;
    }
  return 0;
}

static void
split_fq_name (t_uchar ** archive, t_uchar ** revision, t_uchar * fq_revision)
{
  *archive = arch_parse_package_name (arch_ret_archive, "", fq_revision);
  *revision =  arch_parse_package_name (arch_ret_non_archive, 0 , fq_revision);
}

static void
find_patchlists (rel_table * loc_patches,
		 rel_table * arch_patches,
		 int chatter_fd, 
		 t_uchar * archive, 
	 	 t_uchar * revision,
		 struct arch_archive * arch,
		 assoc_table * contin_list,
		 t_uchar * cache_dir)
{
  t_uchar * lib_descendant_archive = 0;
  t_uchar * lib_descendant_revision = 0;

  rel_table loc_ancestor_patches = 0;
  rel_table descendant_patches = 0;
  t_uchar * fq_loc_ancestor = 0;
  t_uchar * fq_arch_ancestor = 0;
  find_prev_revision (contin_list, &fq_loc_ancestor, &fq_arch_ancestor, chatter_fd, arch, revision, cache_dir);
  if (fq_loc_ancestor)
    {
      t_uchar * loc_ancestor_archive = arch_parse_package_name (arch_ret_archive, "", fq_loc_ancestor);
      t_uchar * loc_ancestor_revision =  arch_parse_package_name (arch_ret_non_archive, 0 , fq_loc_ancestor);
      make_patch_list (chatter_fd, &loc_ancestor_patches, loc_ancestor_archive, loc_ancestor_revision, archive, revision, *contin_list, arch);
      lim_free (0, loc_ancestor_archive);
      lim_free (0, loc_ancestor_revision);
    }
  if (fq_arch_ancestor)
    {
      t_uchar * arch_archive, * arch_revision;
      split_fq_name (&arch_archive, &arch_revision, fq_arch_ancestor);
      make_patch_list (chatter_fd, arch_patches, arch_archive, arch_revision, archive, revision, *contin_list, arch);
      lim_free (0, arch_archive);
      lim_free (0, arch_revision);
    }
  if (find_library_forwards (chatter_fd, &lib_descendant_archive, &lib_descendant_revision, archive, revision))
    {
      if (direct_descendant (arch, revision, lib_descendant_revision))
	make_patch_list (chatter_fd, &descendant_patches, archive, revision, lib_descendant_archive, lib_descendant_revision, *contin_list, arch);
    }
  
  select_patch_list (loc_patches, descendant_patches);
  select_patch_list (loc_patches, loc_ancestor_patches);
}

static int
direct_descendant (struct arch_archive * arch, t_uchar * revision, 
                   t_uchar * candidate_descendant)
{
  t_uchar * newest_cached = 0;
  t_uchar * oldest_cached = 0;
  enum arch_revision_type type;
  int num_scanned;
  int answer;
  t_uchar * last_revision = scan_archive (&type, &num_scanned, &newest_cached, &oldest_cached, arch, candidate_descendant, revision);
  answer = (str_cmp (last_revision, revision) == 0);

  lim_free (0, newest_cached);
  lim_free (0, oldest_cached);
  lim_free (0, last_revision);
  return answer;
}


/*
 * Note that tag-based branches, the previous revision is not an *ancestor*
 */
static t_uchar * 
find_previous_library (int chatter_fd, t_uchar * archive, t_uchar * revision)
{
  int r;
  int answer = 0;
  t_uchar * my_patchlevel = arch_parse_package_name (arch_ret_patch_level, "", revision);
  t_uchar * my_version = arch_parse_package_name (arch_ret_package_version, "", revision);
  rel_table revisions = arch_library_revisions (archive, my_version, 0);
  t_uchar * found_revision = 0;
  

  for (r = r = rel_n_records (revisions)-1; r >=0;  r--)
    {
      if (arch_cmp_revision (revisions[r][0], my_patchlevel) <= 0)
	{
	  answer=1;
	  break;
	}
    }
  if (answer)
    {
      found_revision = str_alloc_cat_many (0, my_version, "--", revisions[r][0], str_end);
      arch_chatter (chatter_fd, "* found previous library revision %s/%s\n", archive, found_revision);
    }

  lim_free(0, my_version);
  lim_free(0, my_patchlevel);
  rel_free_table(revisions);

  return found_revision;
}


/*
 * Note that tag-based branches, the previous revision is not an *ancestor*
 */
static t_uchar *
find_previous_pristine (struct arch_archive *arch, t_uchar * revision, t_uchar * cache_dir)
{
  t_uchar * my_revision = 0;
  t_uchar * answer = 0;
  if (!cache_dir) return answer;
  my_revision = str_save (0, revision);
  while (my_revision)
    {
      t_uchar * prev_revision = 0;
      /****************************************************************
       * Try the pristine cache.
       */
      t_uchar * pristine_copy = arch_find_pristine (0, cache_dir, arch->official_name, my_revision, arch_any_pristine, arch_cache_dir_pristine_search);
      if (pristine_copy)
	{
	  lim_free (0, pristine_copy);
	  answer=str_save (0, my_revision);
	  break;
	}

      /********************************
       * Find the previous revision
       */
      prev_revision = arch_previous_revision (arch, my_revision);
      lim_free (0, my_revision);
      my_revision = prev_revision;
    }

  lim_free (0, my_revision);
  return answer;
}


static int
find_multisource (t_uchar ** lib_rev, 
		  t_uchar ** arch_rev, 
		  t_uchar ** continuation, 
		  int chatter_fd,
		  struct arch_archive * arch, 
		  t_uchar * revision,
		  t_uchar * cache_dir)
{
  int i = 0;
  t_uchar * candidate_revision = 0;
  t_uchar * pristine_candidate = 0;
  t_uchar * last_revision = 0;
  t_uchar * oldest_cached = 0;
  t_uchar * newest_cached = 0;
  enum arch_revision_type type;

  arch_chatter (chatter_fd, "* checking for %s/%s or earlier\n", arch->official_name, revision); 

  if (arch_rev && *arch_rev)
    arch_rev = 0;
  candidate_revision = find_previous_library (chatter_fd, arch->official_name, revision);
  pristine_candidate = find_previous_pristine (arch, revision, cache_dir);
  if (pristine_candidate)
  {
    int use_pristine = 0;
    arch_chatter (chatter_fd, "* found candidate pristine %s/%s\n", arch->official_name, pristine_candidate); 

    if (!candidate_revision)
      use_pristine = 1;
    else
      {
	t_uchar * candidate_level = arch_parse_package_name (arch_ret_patch_level, "", candidate_revision);
	t_uchar * pristine_level = arch_parse_package_name (arch_ret_patch_level, "", pristine_candidate);
	if (arch_cmp_revision (candidate_level, pristine_level) < 0)
	  use_pristine = 1;
      }

    if (use_pristine)
      {
	lim_free (0, candidate_revision);
	candidate_revision = pristine_candidate;
      }
    else
      lim_free (0, pristine_candidate);
  }

  last_revision = scan_archive (&type, &i, &newest_cached, &oldest_cached,
                                arch, revision, candidate_revision);
  if (arch_rev)
    {
      if (newest_cached)
        {
          *arch_rev = newest_cached;
          newest_cached = 0;
        }
      else if (type == arch_import_revision)
        {
          *arch_rev = str_save (0, last_revision);
        }
    }
  if (arch_rev && *arch_rev)
    {
      arch_chatter (chatter_fd, "* found archive revision %s/%s\n", arch->official_name, *arch_rev); 
    }

  if (str_cmp (candidate_revision, last_revision) == 0)
    *lib_rev = candidate_revision;
  else
    {
      lim_free (0, candidate_revision);
      lib_rev = 0;
    }

  if (type == arch_continuation_revision)
    {
      *continuation = str_save (0, last_revision);
    }

  lim_free (0, last_revision);
  lim_free (0, newest_cached);
  lim_free (0, oldest_cached);
  
  return i;
}


static void 
find_prev_revision (assoc_table  * contin_list,
		    t_uchar ** local_rev, 
		    t_uchar ** arch_rev, 
		    int chatter_fd,
		    struct arch_archive * arch, 
		    t_uchar * start_revision,
		    t_uchar * cache_dir)
{
  struct arch_archive * my_arch = arch;
  int patch_n = 0;
  int arch_rev_n = -1;
  t_uchar * my_start_revision = str_save (0, start_revision);
  *arch_rev = 0; 
  while (1)
    {
      t_uchar * fq_prev_revision = 0;
      t_uchar * continuation = 0;
      patch_n += find_multisource (local_rev, (*arch_rev == 0)?arch_rev : 0, &continuation, chatter_fd, my_arch, my_start_revision, cache_dir);
      lim_free (0, my_start_revision);
      my_start_revision = 0;
      if (*local_rev)
	{
	  *local_rev = full_revision (my_arch->official_name, *local_rev);
	}
      /* BUG FOR MULTIPLE CACHEREVS */
      if (*arch_rev && arch_rev_n == -1)
      {
	rel_table arch_rev_patches = 0;
	make_patch_list (chatter_fd, &arch_rev_patches, my_arch->official_name, 
	  *arch_rev, arch->official_name, start_revision, *contin_list, arch);
	arch_rev_n = rel_n_records (arch_rev_patches) - 1;
	*arch_rev = full_revision (my_arch->official_name, *arch_rev);
      }
      if (continuation)
	{
	  t_uchar * fq_key = full_revision (my_arch->official_name, continuation);
	  fq_prev_revision = arch_get_continuation (my_arch, continuation);
	  if (!arch_valid_package_name (fq_prev_revision, arch_req_archive, arch_req_patch_level, 0))
	    {
	      safe_printfmt (2, "arch_build_revision: archive contains bogus CONTINUATION file\n  archive: %s\n  revision: %s\n bogosity: %s\n",
			     my_arch->name, continuation, fq_prev_revision);
	      exit (2);
	    }
	  assoc_set (contin_list, fq_key, fq_prev_revision );
	  lim_free (0, fq_prev_revision);
	  lim_free (0, continuation);
	  fq_prev_revision = assoc_ref (*contin_list, fq_key);
	}

      if (*local_rev)
	  return;

      if (arch_rev_n >= 0 && arch_rev_n + 50 < patch_n)
	return;

      if (fq_prev_revision)
	{
	  t_uchar * prev_archive = arch_parse_package_name (arch_ret_archive, "", fq_prev_revision);
	  my_start_revision = arch_parse_package_name (arch_ret_non_archive, 0, fq_prev_revision);
	  if ( str_cmp (my_arch->official_name, prev_archive) )
	    {
	      if (my_arch != arch)
		arch_archive_close (my_arch);
	      my_arch = archive_connect_soft (prev_archive);
	    }
	  if (!my_arch)
	    arch_chatter (chatter_fd, "* archive %s is not registered\n", prev_archive);
	  lim_free (0, prev_archive);
	}
      if (!my_arch || !my_start_revision)
	break;
    }
  if (my_arch != arch)
    arch_archive_close (my_arch);
}


static t_uchar *
scan_archive(enum arch_revision_type * type,
             int * num_scanned,
             t_uchar ** newest_cached, 
             t_uchar ** oldest_cached, 
	     struct arch_archive *arch, 
	     t_uchar * start_revision, 
	     t_uchar * end_revision)
{
  t_uchar * my_revision = str_save (0, start_revision); 

  *newest_cached = 0;
  *oldest_cached = 0;

  for (*num_scanned = 0; 1; ++*num_scanned)
    {
      int is_cached;
      t_uchar * prev_revision = 0;
      
      arch_revision_type (type, &is_cached, NULL, arch, my_revision);
      if (is_cached)
        {
          if (*newest_cached == 0)
            *newest_cached = str_save (0, my_revision);
          lim_free (0, *oldest_cached);
          *oldest_cached = str_save (0, my_revision);
        }
      if (*type != arch_simple_revision)
        break;
      
      if (str_cmp (my_revision, end_revision) == 0)
        break;
      prev_revision = arch_previous_revision (arch, my_revision);
      lim_free (0, my_revision);
      my_revision = prev_revision;
    }
  return my_revision;
}


static struct arch_archive * archive_connect_soft (t_uchar * archive)
{
   t_uchar *loc = arch_archive_location (archive, 1);
   if (loc == 0)
      return 0;
   lim_free(0, loc);
   return arch_archive_connect_ext (archive, 0, 1);
}


int find_library_forwards (int chatter_fd, t_uchar ** found_archive, t_uchar ** found_revision, t_uchar * archive, t_uchar * revision)
{
  int r;
  int answer = 0;
  t_uchar * my_patchlevel = arch_parse_package_name (arch_ret_patch_level, "", revision);
  t_uchar * my_version = arch_parse_package_name (arch_ret_package_version, "", revision);
  rel_table revisions = arch_library_revisions (archive, my_version, 0);
  
  arch_chatter (chatter_fd, "* checking libraries for  %s/%s--%s or later\n", archive, my_version, my_patchlevel); 
  for (r = 0; r < rel_n_records (revisions);  r++ )
    {
      if (arch_cmp_revision (revisions[r][0], my_patchlevel) >= 0)
	{
	  answer=1;
	  break;
	}
    }
  if (answer)
    {
      *found_archive = str_save (0, archive);
      *found_revision = str_alloc_cat_many (0, my_version, "--", revisions[r][0], str_end);
      arch_chatter (chatter_fd, "* found descendant revision %s/%s\n", *found_archive, *found_revision);
    }

  lim_free(0, my_version);
  lim_free(0, my_patchlevel);
  rel_free_table(revisions);

  return answer;
}


static int 
copy_local_revision (int chatter_fd, 
		     t_uchar * dest_dir,  
		     t_uchar * archive, 
		     t_uchar * revision, 
		     t_uchar * cache_dir,
		     int try_link)
{
  if (copy_pristine_revision (chatter_fd, dest_dir, archive, revision, cache_dir))
    return 3;
  else 
    return copy_library_revision (chatter_fd, dest_dir, archive, revision, try_link);
  return 0; 
}


static int 
copy_library_revision (int chatter_fd, 
              t_uchar * dest_dir, 
	      t_uchar * archive,  
	      t_uchar * revision,
	      int try_link)
{
  rel_table index_by_name = 0;
  t_uchar * library_rev_dir = 0;
  int status = 0;
  arch_patch_id patch_id;
  
  arch_patch_id_init_archive (&patch_id, archive, revision);
  
  library_rev_dir = arch_library_find (0, &patch_id, 1);
  if (!library_rev_dir) return status;
  if (try_link)
    {
      struct stat from, to;
      safe_stat (library_rev_dir, &from);
      safe_stat (dest_dir, &to);
      if (from.st_dev == to.st_dev)
	{
	  arch_chatter (chatter_fd, "* from revision library (linking): %s/%s\n", archive, revision);
	  safe_rmdir (dest_dir);
	  build_link_tree (library_rev_dir, dest_dir);
	  status = 2;
	  goto link_done;
	}
    }
  arch_chatter (chatter_fd, "* from revision library: %s/%s\n", archive, revision);

  index_by_name = arch_library_index (archive, revision);
  rel_sort_table_by_field (0, index_by_name, 0);
  copy_file_list (dest_dir, library_rev_dir, index_by_name);

  rel_free_table (index_by_name);
  status = 1;

  link_done:
  lim_free (0, library_rev_dir);
  arch_patch_id_finalise (&patch_id);
  return status;
}


static int 
copy_pristine_revision (int chatter_fd, 
               t_uchar * dest_dir,  
	       t_uchar * archive, 
	       t_uchar *revision, 
	       t_uchar * cache_dir)
{
  t_uchar * pristine_copy = 0;
  int answer = 0;
  if (!cache_dir) return answer;
  pristine_copy = arch_find_pristine (0, cache_dir, archive, revision, arch_any_pristine, arch_cache_dir_pristine_search);

  if (pristine_copy)
    {
      rel_table file_list = 0;
      answer = 1;

      arch_chatter (chatter_fd, "* from pristine cache: %s/%s\n", archive, revision);

      file_list = arch_source_inventory (pristine_copy, 1, 0, 0);
      copy_file_list (dest_dir, pristine_copy, file_list);

      rel_free_table (file_list);
    }

  lim_free (0, pristine_copy);
  return answer;
}

static void
move_tmp_revision (t_uchar *dest_dir, t_uchar *tmpdir)
{
  int x;
  rel_table files = directory_files (tmpdir);

  for (x = 0; x < rel_n_records (files); ++x)
    {
      if (str_cmp (".", files[x][0]) && str_cmp ("..", files[x][0]))
	{
	  t_uchar * frm = 0;
	  t_uchar * to = 0;

	  frm = file_name_in_vicinity (0, tmpdir, files[x][0]);
	  to = file_name_in_vicinity (0, dest_dir, files[x][0]);

	  safe_rename (frm, to);

	  lim_free (0, frm);
	  lim_free (0, to);
	}
    }

  safe_rmdir (tmpdir);
  rel_free_table (files);
}


static int 
copy_archive_revision(int chatter_fd,
		      t_uchar * dest_dir,
		      struct arch_archive *arch,
		      t_uchar * revision)
{
  enum arch_revision_type type;
  int is_cached;
  arch_revision_type (&type, &is_cached, NULL, arch, revision);
  
  /********************************
   * Archive Cached Revision?
   */
  if (is_cached)
  {
    copy_cached_revision (chatter_fd, dest_dir, arch, revision);
    return 5;
  }

  /********************************
   * Import Revision?
   */
  else if (type == arch_import_revision)
  {
    copy_import_revision (chatter_fd, dest_dir, arch, revision);
    return 4;
  }
  
  else return 0;
}


static void
copy_cached_revision(int chatter_fd, 
		     t_uchar * dest_dir, 
		     struct arch_archive * arch, 
		     t_uchar * revision)
{
  t_uchar * tmpdir = 0;
  arch_chatter (chatter_fd, "* from archive cached: %s/%s\n", arch->official_name, revision);

  tmpdir = tmp_file_name (dest_dir, ",,cached-revision");
  arch_get_cached_revision (arch, revision, tmpdir);
  move_tmp_revision (dest_dir, tmpdir);
  lim_free (0, tmpdir);
}


static void 
copy_import_revision (int chatter_fd, 
		      t_uchar * dest_dir, 
		      struct arch_archive * arch, 
		      t_uchar * revision)
{
  t_uchar * tmpdir = 0;

  arch_chatter (chatter_fd, "* from import revision: %s/%s\n", arch->official_name, revision);

  tmpdir = tmp_file_name (dest_dir, ",,import-revision");
  arch_get_import_revision (arch, revision, tmpdir);
  move_tmp_revision (dest_dir, tmpdir);
  lim_free (0, tmpdir);
}

static t_uchar * 
full_revision (t_uchar * archive, t_uchar * revision)
{
  return str_alloc_cat_many (0, archive, "/", revision, str_end);
}

static int
find_for_previous_revision (int chatter_fd,
                            struct arch_archive ** prev_arch, 
                            t_uchar **prev_archive, 
			    t_uchar **prev_revision, 
			    assoc_table contin_list,
			    struct arch_archive * arch, 
			    t_uchar *archive,
			    t_uchar *revision)
{
  t_uchar * fq_rev = full_revision (arch->official_name, revision);
  t_uchar * fq_prev_revision = assoc_ref (contin_list, fq_rev);
  lim_free (0, fq_rev);
  if (!fq_prev_revision)
    {
      *prev_revision = arch_previous_revision (arch, revision);
      if (! *prev_revision)
	return -1;
      *prev_arch = arch;
      *prev_archive = str_save (0, archive);
      lim_free (0, fq_prev_revision);
    }
  else
    {
      if (!arch_valid_package_name (fq_prev_revision, arch_req_archive, arch_req_patch_level, 0))
	{
	  safe_printfmt (2, "arch_build_revision: continuation list contains bogus entry\n  archive: %s\n  revision: %s\n bogosity: %s\n",
			 archive, revision, fq_prev_revision);
	  exit (2);
	}
      *prev_archive = arch_parse_package_name (arch_ret_archive, 0, fq_prev_revision);
      *prev_revision = arch_parse_package_name (arch_ret_non_archive, 0, fq_prev_revision);
      
      *prev_arch = archive_connect_soft (*prev_archive);
      if ( *prev_arch == 0)
	return -1;
      
    }
  return 0;
}


static void
apply_revision_patch (int chatter_fd, 
		      t_uchar * dest_dir, 
		      struct arch_archive * arch, 
		      t_uchar * revision,
		      int reverse)
{
  t_uchar * tmppatch = 0;
  struct arch_apply_changeset_report r;
  


  tmppatch = tmp_file_name (dest_dir, ",,next-patch");
  arch_get_patch (arch, revision, tmppatch);

  mem_set0 ((t_uchar *)&r, sizeof (r));

  arch_apply_changeset (&r, tmppatch, dest_dir, arch_unspecified_id_tagging, arch_inventory_unrecognized, reverse, 0, 0);

  if (arch_conflicts_occurred (&r))
    panic ("conflict applying patch in arch_build_revision");

  arch_free_apply_changeset_report_data (&r);

  rmrf_file (tmppatch);
  lim_free (0, tmppatch);
}

static void 
apply_revisions (int chatter_fd, 
                 t_uchar * dest_dir, 
		 struct arch_archive * arch, 
		 rel_table patch_list,
		 int apply_direction)
{
  int i;
  struct arch_archive * my_arch = arch;
  int reverse = (apply_direction == 1) ? 0 : 1;
  int start;
  int end;

  if (apply_direction == 1)
  {
     start = 1;
     end = rel_n_records (patch_list);
  }
  else
  {
     start = rel_n_records(patch_list) -1;
     end = 0;
  }
  for (i=start; i != end; i +=apply_direction)
  {
    if (my_arch == 0 || str_cmp(my_arch->official_name, patch_list[i][0]))
      {
	arch_chatter (chatter_fd, "* opening archive: %s\n", patch_list[i][0]);
	if (my_arch != arch)
	  arch_archive_close (my_arch);
	my_arch = arch_archive_connect (patch_list[i][0], 0);
      }

    if (!reverse) arch_chatter (chatter_fd, "* patching to revision: %s/%s\n", my_arch->official_name, patch_list[i][1]);
    else arch_chatter (chatter_fd, "* (reverse) patching to revision: %s/%s\n", my_arch->official_name, patch_list[i-1][1]);
    apply_revision_patch (chatter_fd, dest_dir, my_arch, patch_list[i][1], reverse);
  }

  if (my_arch != arch)
    arch_archive_close (my_arch);
}


/* tag: Tom Lord Wed May 21 15:59:22 2003 (build-revision.c)
 */
