/* cmd-library-remove.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "libfsutils/rmrf.h"
#include "libarch/my.h"
#include "libarch/libraries.h"
#include "libarch/namespace.h"
#include "commands/library-remove.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] revision");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_archive, "A", "archive", 1, \
      N_("Override `my-default-archive'"))

t_uchar arch_cmd_library_remove_help[] = N_("remove a revision from the revision library\n"
                                          "Remove REVISION from your revision library.\n"
                                          "\n"
                                          "If the revision is not present, print an error message\n"
                                          "and exit with non-0 status.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_library_remove (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * default_archive;

  default_archive = 0;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_library_remove_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;


        case opt_archive:
          {
            default_archive = str_save (0, option->arg_string);
            break;
          }
        }
    }

  if (argc != 2)
    goto usage_error;

  {
    t_uchar * revspec;
    t_uchar * archive = 0;
    t_uchar * revision = 0;
    t_uchar * answer;
    arch_patch_id patch_id;

    revspec = argv[1];

    if (!arch_valid_package_name (revspec, arch_maybe_archive, arch_req_patch_level, 0))
      {
        safe_printfmt (2, "%s: invalid revision name (%s)\n",
                       argv[0], revspec);
        exit (2);
      }

    archive = arch_parse_package_name (arch_ret_archive, default_archive, revspec);
    if (!archive)
      {
        safe_printfmt (2, "%s: no archive specified for library-remove\n",
                       argv[0]);
        exit (2);
      }

    revision = arch_parse_package_name (arch_ret_non_archive, 0, revspec);
    /* should patch ID provide query methods to replace the above two if blocks ? */
    arch_patch_id_init_archive (&patch_id, archive, revision);

#undef FIXME
    /* way to ask for particular device or particular library?
     */
    answer = arch_library_find (0, &patch_id, 0);

    arch_patch_id_finalise (&patch_id);


    if (answer)
      rmrf_file (answer);
    else
      {
        safe_printfmt (2, "%s: revision not present in library (%s/%s)\n",
                       argv[0], archive, revision);
        exit (1);
      }
  }

  return 0;
}




/* tag: Tom Lord Wed May 21 15:44:27 2003 (library-remove.c)
 */
