// vs_text_layout.cc
//
//  Copyright 2004 Daniel Burrows

#include "vs_text_layout.h"

#include "fragment.h"
#include "fragment_contents.h"
#include "vscreen.h"

#include "config/keybindings.h"

#include <algorithm>

#include <sigc++/object_slot.h>

using namespace std;

keybindings *vs_text_layout::bindings;

vs_text_layout::vs_text_layout():start(0), f(newline_fragment()), stale(true), lastw(0)
{
  do_layout.connect(slot(*this, &vs_text_layout::layout_me));
}

vs_text_layout::vs_text_layout(fragment *_f):start(0), f(_f), stale(true), lastw(0)
{
  do_layout.connect(slot(*this, &vs_text_layout::layout_me));
}

void vs_text_layout::init_bindings()
{
  bindings=new keybindings(&global_bindings);
}

bool vs_text_layout::handle_char(chtype ch)
{
  if(bindings->key_matches(ch, "Up"))
    line_up();
  else if(bindings->key_matches(ch, "Down"))
    line_down();
  else if(bindings->key_matches(ch, "Begin"))
    move_to_top();
  else if(bindings->key_matches(ch, "End"))
    move_to_bottom();
  else if(bindings->key_matches(ch, "PrevPage"))
    page_up();
  else if(bindings->key_matches(ch, "NextPage"))
    page_down();
  else
    return vscreen_widget::handle_char(ch);

  return true;
}

size vs_text_layout::size_request()
{
  return size(1,1);
}

vs_text_layout::~vs_text_layout()
{
  delete f;
}

void vs_text_layout::set_fragment(fragment *_f)
{
  delete f;
  f=_f;

  stale=true;

  vscreen_update();
}

void vs_text_layout::set_start(unsigned int new_start)
{
  if(new_start!=start)
    {
      start=new_start;
      do_signal();
      vscreen_update();
    }
}

void vs_text_layout::layout_me()
{
  // If the width has changed, we need to recalculate the layout.
  if(getmaxx()!=lastw)
    stale=true;
}

bool vs_text_layout::get_cursorvisible()
{
  return true;
}

/** The cursor is always located in the upper-left-hand corner. */
point vs_text_layout::get_cursorloc()
{
  return point(0,0);
}

/** This widget can get focus if it can scroll: ie, if its contents take
 *  up more lines than it was allocated.
 */
bool vs_text_layout::focus_me()
{
  freshen_contents();

  if(start>0 || contents.size()>(unsigned) getmaxy())
    return true;
  else
    return false;
}

/** Paint by refreshing the contents [if necessary], then drawing,
 *  starting from the current line.
 */
void vs_text_layout::paint()
{
  freshen_contents();

  if(start>=contents.size())
    {
      if(contents.size()==0)
	set_start(0);
      else
	set_start(contents.size()-1);
    }

  for(int i=0; i<getmaxy() && i+start<contents.size(); ++i)
    mvaddnstr(i, 0, contents[i+start], contents[i+start].size());
}

void vs_text_layout::freshen_contents()
{
  if(stale)
    {
      contents=f->layout(getmaxx());

      do_signal();

      stale=false;
    }
}

void vs_text_layout::line_down()
{
  freshen_contents();

  if(start+getmaxy()<contents.size())
    set_start(start+1);
}

void vs_text_layout::line_up()
{
  freshen_contents();

  if(start>0)
    set_start(start-1);
}

unsigned int vs_text_layout::height_for_width(unsigned int w)
{
  return f->layout(w).size();
}

void vs_text_layout::move_to_top()
{
  set_start(0);
}

void vs_text_layout::move_to_bottom()
{
  freshen_contents();

  set_start(max(start, contents.size()-getmaxy()));
}

void vs_text_layout::page_up()
{
  if(start<(unsigned) getmaxy())
    set_start(0);
  else
    set_start(start-getmaxy());
}

void vs_text_layout::page_down()
{
  freshen_contents();

  if(start+getmaxy()<contents.size())
    set_start(start+getmaxy());
}

// Assumes the contents are already fresh.
void vs_text_layout::do_signal()
{
  if(((unsigned) getmaxy())>=contents.size() && start==0)
    location_changed(start, 0);
  else if(start+getmaxy()>=contents.size())
    location_changed(1, 1);
  else
    location_changed(start, contents.size()-getmaxy());
}

void vs_text_layout::scroll(bool dir)
{
  if(dir)
    page_up();
  else
    page_down();
}
