// vs_pager.h       -*-c++-*-
//
//  Copyright 2000-2004 Daniel Burrows
//
//  A vscreen which acts as a text pager.  Recently rewritten to use
//  an internal vector of lines (makes its behavior much nicer from
//  the user's point of view, even if it's slightly less efficient).

#ifndef VS_PAGER_H
#define VS_PAGER_H

#include "vscreen_widget.h"

#include <string>
#include <vector>

class keybindings;

class vs_pager:public vscreen_widget
{
public:
  typedef std::vector<std::string>::size_type line_count;
  typedef std::string::size_type col_count;
private:
  /** The lines of text being displayed: */
  std::vector<std::string> lines;

  /** The first visible line. */
  line_count first_line;

  /** How much the widget is scrolled to the right. */
  col_count first_column;

  /** The bounds of the text contained in the pager. */
  col_count text_width;

  /** The last string the user searched for (so we can repeat searches) */
  std::string last_search;

  /** Handles resizing the widget. */
  void layout_me();
public:
  /** Create a vs_pager from the given memory region.
   *
   *  \param text the text to display
   *  \param len the length of the buffer
   */
  vs_pager(const char *text, int len);

  /** Create a vs_pager from a string.
   *
   *  \param s the text to display
   */
  vs_pager(const std::string &s);

  /** Destroy this vs_pager. */
  virtual ~vs_pager();

  /** Set the text to the given memory region.
   *
   *  \param text the text to display
   *  \param len the length of the buffer
   */
  virtual void set_text(const char *text, std::string::size_type len);

  /** Change the displayed text. */
  virtual void set_text(const std::string &s);

  /** Scroll the screen up by the given number of lines. */
  void scroll_up(line_count nlines);

  /** Scroll the screen down by the given number of lines. */
  void scroll_down(line_count nlines);

  /** Scroll the screen right by the given number of columns. */
  void scroll_right(col_count ncols);

  /** Scroll the screen left by the given number of columns. */
  void scroll_left(col_count ncols);

  /** Scroll to the top of the screen. */
  void scroll_top();

  /** Scroll to the bottom of the screen. */
  void scroll_bottom();

  /** Scroll by a page in the given direction.
   *
   *  \param dir if \b true, scroll a page up; otherwise, scroll a
   *  page down.
   */
  void scroll_page(bool dir);

  /** Find the next line containing the given string.
   *
   *  \param s the string to search for
   */
  void search_for(std::string s);

  /** Return the last string which the user searched for. */
  std::string get_last_search() {return last_search;}

  line_count get_first_line() {return first_line;}
  line_count get_num_lines() {return lines.size();}
  col_count get_first_column() {return first_column;}
  col_count get_num_columns() {return text_width;}

  /** Emits a signal describing the verical location of the display
   *  within the text.
   */
  void do_line_signal();

  /** Emits a signal describing the horizontal location of the display
   *  within the text.
   */
  void do_column_signal();

  // vscreen overrides:

  virtual bool handle_char(chtype ch);
  virtual bool focus_me() {return true;}
  virtual void paint();

  size size_request();
  bool get_cursorvisible() {return true;}
  point get_cursorloc() {return point(0,0);}

  /** Announces that the user has scrolled vertically. */
  SigC::Signal2<void, int, int> line_changed;

  /** Announces that the user has scrolled horizontally. */
  SigC::Signal2<void, int, int> column_changed;

  static keybindings *bindings;
  static void init_bindings();
};

class vs_file_pager:public vs_pager
{
public:
  vs_file_pager();
  vs_file_pager(std::string filename);
  vs_file_pager(const char *text, int len);

  void load_file(std::string filename);
};

#endif
