/* AbiSource Program Utilities
 * 
 * Copyright (C) 2002 Dom Lachowicz <cinamod@hotmail.com>
 * Copyright (C) 2005 Daniel d'Andrada T. de Carvalho
 * <daniel.carvalho@indt.org.br>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */
 
// Class definition include
#include "OD_Style_PageLayout.h"

// Internal includes
#include "OD_ListenerStateAction.h"

// AbiWord includes
#include <ut_math.h>
#include <fp_PageSize.h>
#include <pd_Document.h>





/**
 * 
 */
void OD_Style_PageLayout::startElement(const XML_Char* pName,
                                       const XML_Char** ppAtts,
                                       OD_ListenerStateAction& rAction) {

    const XML_Char* pVal;
                                        
    if (!UT_strcmp("style:page-layout", pName)) {
        
        pVal = UT_getAttribute ("style:name", ppAtts);
        UT_ASSERT(pVal);
        m_name = pVal;
        
    } else if (!UT_strcmp("style:page-layout-properties", pName)) {
        
        _parse_pageLayoutProperties(ppAtts);
        
    } else if (!UT_strcmp("style:columns", pName)) {
        
        pVal = UT_getAttribute ("fo:column-count", ppAtts);
        if (pVal) {
            m_columnCount = UT_UTF8String_sprintf("columns: %s", pVal);
        }
        
        pVal = UT_getAttribute ("fo:column-gap", ppAtts);
        if (pVal) {
            m_columnGap = UT_UTF8String_sprintf("column-gap: %s", pVal);
        }
    }
}





/**
 * 
 */
void OD_Style_PageLayout::endElement(const XML_Char* pName,
                                     OD_ListenerStateAction& rAction) {
                                        
    if (!UT_strcmp("style:page-layout", pName)) {
        _buildSectionPropsString();
        rAction.popState();
    }
}





/**
 * Parses the <style:page-layout-properties> start tag.
 */
void OD_Style_PageLayout::_parse_pageLayoutProperties(const XML_Char** ppAtts) {
    
    const XML_Char* pVal = NULL;
    UT_uint32 propCtr = 0;
    
    m_pageWidthNumeric = 0;
    m_pageHeightNumeric = 0;

    // will go into the <pagesize> tag
    pVal = UT_getAttribute ("fo:page-width", ppAtts);
    if (pVal) {
        m_pageWidthNumeric = rint(UT_convertToDimension(pVal, DIM_MM));
        m_pageWidth = UT_UTF8String_sprintf("%f", m_pageWidthNumeric);
    }

    pVal = UT_getAttribute ("fo:page-height", ppAtts);
    if (pVal) {
        m_pageHeightNumeric = rint(UT_convertToDimension(pVal, DIM_MM));
        m_pageHeight = UT_UTF8String_sprintf("%f", m_pageHeightNumeric);
    }

    pVal = UT_getAttribute ("style:print-orientation", ppAtts);
    if (pVal) {
        m_printOrientation = strdup (pVal);
    }


    // will go as pProps into the <section> tag
    pVal = UT_getAttribute ("fo:margin-left", ppAtts);
    if (pVal)
        m_marginLeft = UT_UTF8String_sprintf("page-margin-left: %s", pVal);

    pVal = UT_getAttribute ("fo:margin-top", ppAtts);
    if (pVal)
        m_marginTop = UT_UTF8String_sprintf("page-margin-top: %s", pVal);

    pVal = UT_getAttribute ("fo:margin-right", ppAtts);
    if (pVal)
        m_marginRight = UT_UTF8String_sprintf("page-margin-right: %s", pVal);

    pVal = UT_getAttribute ("fo:margin-bottom", ppAtts);
    if (pVal)
        m_marginBottom = UT_UTF8String_sprintf("page-margin-bottom: %s", pVal);
}





/**
 * Sets the <pagesize> tag.
 */
void OD_Style_PageLayout::definePageSizeTag(PD_Document* pDocument) const {
    
    UT_uint32 propCtr = 0;
    static const int MAX_PAGE_ATTS = 13; // 2*(width height orientation pagetype units page-scale) 0 
    const XML_Char* pageAtts[MAX_PAGE_ATTS];

    if (!m_pageWidth.empty()) {
        pageAtts[propCtr++] = "width";
        pageAtts[propCtr++] = m_pageWidth.utf8_str();
    }

    if (!m_pageHeight.empty()) {
        pageAtts[propCtr++] = "height";
        pageAtts[propCtr++] = m_pageHeight.utf8_str();
    }

    pageAtts[propCtr++] = "units";
    pageAtts[propCtr++] = "mm";

    if (!m_printOrientation.empty()) {
        pageAtts[propCtr++] = "orientation";
        pageAtts[propCtr++] = UT_strdup (m_printOrientation.utf8_str());
    }

    pageAtts[propCtr++] = "page-scale";
    pageAtts[propCtr++] = "1.0";

    // width and height are rounded to full mm because that's how they are
    // predefined in Abi and there seem to be rounding errors in oow's exporter
    fp_PageSize ps(m_pageWidthNumeric, m_pageHeightNumeric, DIM_MM);
    pageAtts[propCtr++] = "pagetype";
    pageAtts[propCtr++] = ps.getPredefinedName();

    pageAtts[propCtr] = 0; // To signal the end of the array
    
    
    pDocument->setPageSizeFromFile(pageAtts);
}


/**
 * 
 */
void OD_Style_PageLayout::_buildSectionPropsString() {
    
        m_sectionProps.clear();
#define APPEND_STYLE(sty) if (sty.size()) {if(m_sectionProps.size()){m_sectionProps += "; ";}m_sectionProps += sty;}
        APPEND_STYLE(m_marginLeft);
        APPEND_STYLE(m_marginTop);
        APPEND_STYLE(m_marginRight);
        APPEND_STYLE(m_marginBottom);
        APPEND_STYLE(m_columnCount);
        APPEND_STYLE(m_columnGap);
#undef APPEND_STYLE

}
